/**
 * \file CPageTextElementAlternative.h
 * \brief Interface for CPageTextElementAlternative class
 * 
 * \internal
 *
 * 2023-10-20 - IRIS SDK
 * - iDRS 16.1.0 release
 *
 * \endinternal
 */
 
#ifndef _CPageTextElementAlternative_h_
#define _CPageTextElementAlternative_h_

#include "TObjPtrArray.h"
#include "CPageCustomData.h"
#include "EnumDefinitions.h"

namespace IDRS
{

  class CPageTextElementAlternativeData;
  class idrs_string;
  /**
   * \brief CPageTextElementAlternative represents a text element alternative
   *
   * Text alternatives are usually generated by the ocr engine with text solutions close to the actual solution.
   * Each text alternative has it's own text solution and confidence level
   */
  class IDRSKRN_API CPageTextElementAlternative : public CPageCustomData
  {
  public:
    /**
     * \brief Default constructor
     */
    inline CPageTextElementAlternative();
    /**
     * \brief Copy constructor
     *
     * \param argPageTextAlternative Object to copy (shallow copy)
     */
    inline CPageTextElementAlternative(const CPageTextElementAlternative& argPageTextAlternative);
    /**
     * \brief CPageTextElementAlternative constructor that initializes an object with a data object
     *
     * \see CObjPtr and CObj for iDRS SDK smart pointers and data classes
     *
     * \param pObj Pointer to an object representing the data class
     */
    inline CPageTextElementAlternative(CPageTextElementAlternativeData * pObj);
    /**
     * \brief CPageTextElementAlternative constructor that initializes an object with a data object
     *
     * \see CObjPtr and CObj for iDRS SDK smart pointers and data classes
     *
     * \param pObj Pointer to an object representing the data class
     */
    inline CPageTextElementAlternative(CObj * pObj);
    /**
     * \brief Destructor
     */
    inline virtual ~CPageTextElementAlternative();
    /**
     * \brief Creates a new instance of CPageTextElementAlternative
     */
    static CPageTextElementAlternative Create();
    /**
     * \brief Retrieves the text value for the alternative
     */
    const idrs_string& GetTextValue() const;
    /**
     * \brief Sets the text value for the alternative
     *
     * \param strTextValue The text value for the alternative
     */
    void SetTextValue(const idrs_string& strTextValue);
    /**
     * \brief Retrieves the confidence value for the text element.
     *
     * The confidence value ranges from 0 (lowest confidence) to 100 (highest confidence).
     *
     * This value corresponds to the likeliness that the provided result is correct, and is computed by the OCR engine based on the OCR local score
     * as well as contextual information like surrounding characters, typographical analysis, or presence of the word it belongs to in a lexicon.
     *
     * \return Confidence value for the text element.
     *
     * \remarks In most cases, an integrator who wishes to assess the quality of a text element's recognition should rely on the OCR confidence value instead of
     * on the OCR local score.
     */
    IDRS_UINT GetOcrConfidence() const;
    /**
     * \brief Sets the confidence value for the text element.
     *
     * The confidence value ranges from 0 (lowest confidence) to 100 (highest confidence).
     *
     * This value corresponds to the likeliness that the provided result is correct, and is computed by the OCR engine based on the OCR local score
     * as well as contextual information like surrounding characters, typographical analysis, or presence of the word it belongs to in a lexicon.
     *
     * \param uiOcrConfidence Confidence value for the text element.
     *
     * \remarks In most cases, an integrator who wishes to assess the quality of a text element's recognition should rely on the OCR confidence value instead of
     * on the OCR local score.
     */
    void SetOcrConfidence(const IDRS_UINT uiOcrConfidence);
    /**
     * \brief Retrieves the OCR local score of the text element.
     *
     * The OCR score ranges from 0 (lowest score) to 100 (highest score).
     *
     * This value corresponds to the recognition score of the text element, without taking into account information from surrounding elements.
     * The score corresponds to the result of looking only at the shape of the current character.
     *
     * \return OCR local score for the text element.
     *
     * \remarks In most cases, an integrator who wishes to assess the quality of a text element's recognition should rely on the OCR confidence value instead of
     * on the OCR local score.
     */
    IDRS_UINT GetOcrLocalScore() const;
    /**
     * \brief Sets the OCR local score of the text element.
     *
     * The OCR score ranges from 0 (lowest score) to 100 (highest score).
     *
     * This value corresponds to the recognition score of the text element, without taking into account information from surrounding elements.
     * The score corresponds to the result of looking only at the shape of the current character.
     *
     * \param uiOcrLocalScore OCR local score for the text element.
     *
     * \remarks In most cases, an integrator who wishes to assess the quality of a text element's recognition should rely on the OCR confidence value instead of
     * on the OCR local score.
     */
    void SetOcrLocalScore(const IDRS_UINT uiOcrLocalScore);
  private:
    //! Internal use 
    CPageTextElementAlternativeData & TypedObject() const
    {
      return (CPageTextElementAlternativeData &)Object();
    };

  };

  typedef TObjPtrArray<CPageTextElementAlternative> CPageTextElementAlternativeArray;

  /*
   *
   */
  inline CPageTextElementAlternative::CPageTextElementAlternative()
  {
  }
  /*
   *
   */
  inline CPageTextElementAlternative::CPageTextElementAlternative(const CPageTextElementAlternative& argPageTextAlternative)
    : CPageCustomData((CPageCustomData&)argPageTextAlternative)
  {
  }
  /*
   *
   */
  inline CPageTextElementAlternative::CPageTextElementAlternative(CPageTextElementAlternativeData * pObj)
    : CPageCustomData((CPageCustomDataData*)pObj)
  {
  }  
  /*
   *
   */
  inline CPageTextElementAlternative::CPageTextElementAlternative(CObj * pObj)
    : CPageCustomData(pObj)
  {
  }
  /*
   *
   */
  inline CPageTextElementAlternative::~CPageTextElementAlternative()
  {
  }
}
#endif // _CPageTextElementAlternative_h_
