//---------------------------------------------------------------------------------------
// Copyright (c) 2001-2025 by Apryse Software Inc. All Rights Reserved.
// Consult legal.txt regarding legal and license information.
//---------------------------------------------------------------------------------------

#import <OBJC/PDFNetOBJC.h>
#import <Foundation/Foundation.h>

//---------------------------------------------------------------------------------------
// The following sample illustrates how to convert HTML pages to PDF format using
// the PTHTML2PDF class.
// 
// 'pdftron.PDF.PTHTML2PDF' is an optional PDFNet Add-On utility class that can be 
// used to convert HTML web pages into PDF documents by using an external module (html2pdf).
//
// html2pdf modules can be downloaded from http://www.pdftron.com/pdfnet/downloads.html.
//
// Users can convert HTML pages to PDF using the following operations:
// - Simple one line static method to convert a single web page to PDF. 
// - Convert HTML pages from URL or string, plus optional table of contents, in user defined order. 
// - Optionally configure settings for proxy, images, java script, and more for each HTML page. 
// - Optionally configure the PDF output, including page size, margins, orientation, and more. 
// - Optionally add table of contents, including setting the depth and appearance.
//---------------------------------------------------------------------------------------

int main(int argc, char * argv[])
{
    @autoreleasepool {

        int ret = 0;

        // The first step in every application using PDFNet is to initialize the 
        // library and set the path to common PDF resources. The library is usually 
        // initialized only once, but calling Initialize() multiple times is also fine.
        [PTPDFNet Initialize: 0];

        // For PTHTML2PDF we need to locate the PTHTML2PDF module. If placed with the 
        // PDFNet library, or in the current working directory, it will be loaded
        // automatically. Otherwise, it must be set manually using PTHTML2PDF.SetModulePath.
        [PTHTML2PDF SetModulePath: @"../../../Lib"];
        if( ![PTHTML2PDF IsModuleAvailable] )
        {
            NSLog(@"\n");
            NSLog(@"Unable to run HTMLPDFTest: Apryse SDK HTML2PDF module not available.");
            NSLog(@"---------------------------------------------------------------");
            NSLog(@"The HTML2PDF module is an optional add-on, available for download");
            NSLog(@"at https://www.pdftron.com/. If you have already downloaded this");
            NSLog(@"module, ensure that the SDK is able to find the required files");
            NSLog(@"using the HTML2PDF::SetModulePath() function.");
            return 1;
        }
    
        //--------------------------------------------------------------------------------
        // Example 1) Simple conversion of a web page to a PDF doc. 

        @try
        {
            PTPDFDoc *doc = [[PTPDFDoc alloc] init];
            PTHTML2PDF *converter = [[PTHTML2PDF alloc] init];
            [converter InsertFromURL: @"https://docs.apryse.com/"];
        
            // now convert a web page, sending generated PDF pages to doc
            [converter Convert: doc];
            [doc SaveToFile: @"../../TestFiles/Output/html2pdf_example_01.pdf" flags: e_ptlinearized];
        }
        @catch (NSException *e)
        {
            NSLog(@"%@", e.reason);
            ret = 1;
        }

        //--------------------------------------------------------------------------------
        // Example 2) Modify the settings of the generated PDF pages and attach to an
        // existing PDF document. 

        @try
        {
            // open the existing PDF, and initialize the security handler
            PTPDFDoc *doc = [[PTPDFDoc alloc] initWithFilepath: @"../../TestFiles/numbered.pdf"];
            [doc InitSecurityHandler];

            // create the PTHTML2PDF converter object and modify the output of the PDF pages
            PTHTML2PDF *converter = [[PTHTML2PDF alloc] init];
            [converter SetPaperSize: e_11x17];

            // insert the web page to convert
            [converter InsertFromURL: @"https://docs.apryse.com/"];

            // convert the web page, appending generated PDF pages to doc
            [converter Convert: doc];
            [doc SaveToFile: @"../../TestFiles/Output/html2pdf_example_02.pdf" flags: e_ptlinearized];
        }
        @catch (NSException *e)
        {
            NSLog(@"%@", e.reason);
            ret = 1;
        }

        //--------------------------------------------------------------------------------
        // Example 3) Convert multiple web pages
        @try
        {
            // convert page 0 into pdf
            PTPDFDoc *doc = [[PTPDFDoc alloc] init];
            PTHTML2PDF *converter = [[PTHTML2PDF alloc] init];
            NSString *header = @"<div style='width:15%;margin-left:0.5cm;text-align:left;font-size:10px;color:#0000FF'><span class='date'></span></div><div style='width:70%;direction:rtl;white-space:nowrap;overflow:hidden;text-overflow:clip;text-align:center;font-size:10px;color:#0000FF'><span>PDFTRON HEADER EXAMPLE</span></div><div style='width:15%;margin-right:0.5cm;text-align:right;font-size:10px;color:#0000FF'><span class='pageNumber'></span> of <span class='totalPages'></span></div>";
            NSString *footer = @"<div style='width:15%;margin-left:0.5cm;text-align:left;font-size:7px;color:#FF00FF'><span class='date'></span></div><div style='width:70%;direction:rtl;white-space:nowrap;overflow:hidden;text-overflow:clip;text-align:center;font-size:7px;color:#FF00FF'><span>PDFTRON FOOTER EXAMPLE</span></div><div style='width:15%;margin-right:0.5cm;text-align:right;font-size:7px;color:#FF00FF'><span class='pageNumber'></span> of <span class='totalPages'></span></div>";
            [converter SetHeader:header];
            [converter SetFooter:footer];
            [converter SetMargins : @"1cm"  bottom:@"2cm" left:@".5cm" right:@"1.5cm"];
            PTWebPageSettings *settings = [[PTWebPageSettings alloc] init];
            [settings SetZoom : 0.5];
            [converter InsertFromURLWithSettings: @"https://docs.apryse.com/" settings: settings];
            [converter Convert:doc];

            // convert page 1 with the same settings, appending generated PDF pages to doc
            [converter InsertFromURLWithSettings: @"https://docs.apryse.com/all-products/" settings: settings];
            [converter Convert:doc];

            // convert page 2 with different settings, appending generated PDF pages to doc
            PTHTML2PDF *another_converter = [[PTHTML2PDF alloc] init];
            [another_converter SetLandscape : YES];

            PTWebPageSettings *another_settings = [[PTWebPageSettings alloc] init];
            [another_settings SetPrintBackground : NO];

            [another_converter InsertFromURLWithSettings: @"https://docs.apryse.com/documentation/web/faq" settings: another_settings];
            [another_converter Convert:doc];

            [doc SaveToFile: @"../../TestFiles/Output/html2pdf_example_03.pdf" flags: e_ptlinearized];
        }
        @catch (NSException *e)
        {
            NSLog(@"%@", e.reason);
            ret = 1;
        }

        //--------------------------------------------------------------------------------
        // Example 4) Convert HTML string to PDF. 

        @try
        {
            PTPDFDoc *doc = [[PTPDFDoc alloc] init];
            PTHTML2PDF *converter = [[PTHTML2PDF alloc] init];
        
            // Our HTML data
            NSString *html = @"<html><body><h1>Heading</h1><p>Paragraph.</p></body></html>";
            
            // Add html data
            [converter InsertFromHtmlString: html];
            // Note, InsertFromHtmlString can be mixed with the other Insert methods.
            
            [converter Convert: doc];
            [doc SaveToFile: @"../../TestFiles/Output/html2pdf_example_04.pdf" flags: e_ptlinearized];
        }
        @catch (NSException *e)
        {
            NSLog(@"%@", e.reason);
            ret = 1;
        }

        //--------------------------------------------------------------------------------
        // Example 5) Set the location of the log file to be used during conversion. 

        @try
        {
            PTPDFDoc *doc = [[PTPDFDoc alloc] init];
            PTHTML2PDF *converter = [[PTHTML2PDF alloc] init];
            [converter SetLogFilePath: @"../../TestFiles/Output/html2pdf.log"];
            [converter InsertFromURL: @"https://docs.apryse.com/"];
            [converter Convert: doc];
            [doc SaveToFile: @"../../TestFiles/Output/html2pdf_example_05.pdf" flags: e_ptlinearized];
        }
        @catch (NSException *e)
        {
            NSLog(@"%@", e.reason);
            ret = 1;
        }

        [PTPDFNet Terminate: 0];
        return ret;
    }
}
