//---------------------------------------------------------------------------------------
// Copyright (c) 2001-2025 by Apryse Software Inc. All Rights Reserved.
// Consult legal.txt regarding legal and license information.
//---------------------------------------------------------------------------------------

#import <OBJC/PDFNetOBJC.h>
#import <Foundation/Foundation.h>

// This sample illustrates the basic text highlight capabilities of PDFNet.
// It simulates a full-text search engine that finds all occurrences of the word 'Federal'.
// It then highlights those words on the page.
//
// Note: The TextSearch class is the preferred solution for searching text within a single
// PDF file. TextExtractor provides search highlighting capabilities where a large number
// of documents are indexed using a 3rd party search engine.

int main(int argc, char *argv[])
{
    @autoreleasepool {
        int ret = 0;
        [PTPDFNet Initialize: 0];

        // Sample code showing how to use high-level text highlight APIs.
        @try
        {
            PTPDFDoc *doc = [[PTPDFDoc alloc] initWithFilepath: @"../../TestFiles/paragraphs_and_tables.pdf"];
            [doc InitSecurityHandler];

            PTPage *page = [doc GetPage: 1];
            if (![page IsValid])
            {
                [PTPDFNet Terminate: 0];
                NSLog(@"Page not found.");
                return 1;
            }

            PTTextExtractor *txt = [[PTTextExtractor alloc] init];
            [txt Begin: page clip_ptr: 0 flags: 0]; // Read the page.


            // Retrieve the page text
            // Do not dehyphenate; that would interfere with character offsets
            NSString *page_text = [txt GetAsText: NO];

            // Simulating a full-text search engine that finds all occurrences of the word 'Federal'.
            // In a real application, plug in your own search engine here.
            NSString *search_text = @"Federal";
            NSRange range = [page_text rangeOfString:search_text options:0];
            NSMutableArray<PTTextExtractorCharRange *> *char_ranges = [NSMutableArray<PTTextExtractorCharRange *> array];
            while (range.location != NSNotFound)
            {
                PTTextExtractorCharRange *char_range = [[PTTextExtractorCharRange alloc] init];
                [char_range setIndex: (int)range.location];
                [char_range setLength: (int)range.length];
                [char_ranges addObject: char_range]; // character offset + length
                range = [page_text rangeOfString:search_text options:0 range:NSMakeRange(range.location + 1, page_text.length - range.location - 1)];
            }

            // Retrieve Highlights object and apply annotations to the page
            PTHighlights *hlts = [txt GetHighlights: char_ranges];
            [hlts Begin: doc];
            while ([hlts HasNext])
            {
                PTVectorQuadPoint *quads = [hlts GetCurrentQuads];
                int i = 0;
                for (; i < [quads size]; ++i)
                {
                    // assume each quad is an axis-aligned rectangle
                    PTQuadPoint *q = [quads get: i];
                    double x1 = MIN(MIN(MIN([[q getP1] getX], [[q getP2] getX]), [[q getP3] getX]), [[q getP4] getX]);
                    double x2 = MAX(MAX(MAX([[q getP1] getX], [[q getP2] getX]), [[q getP3] getX]), [[q getP4] getX]);
                    double y1 = MIN(MIN(MIN([[q getP1] getY], [[q getP2] getY]), [[q getP3] getY]), [[q getP4] getY]);
                    double y2 = MAX(MAX(MAX([[q getP1] getY], [[q getP2] getY]), [[q getP3] getY]), [[q getP4] getY]);
                    PTPDFRect *rect = [[PTPDFRect alloc] initWithX1: x1 y1: y1 x2: x2 y2: y2];
                    PTHighlightAnnot *highlight = [PTHighlightAnnot Create: [doc GetSDFDoc] pos: rect];
                    [page AnnotPushBack: highlight];

                    printf("[%.2f, %.2f, %.2f, %.2f]\n", x1, y1, x2, y2);
                }
                [hlts Next];
            }

            // Output highlighted PDF doc
            [doc SaveToFile: @"../../TestFiles/Output/search_highlights.pdf" flags: e_ptlinearized];

        }
        @catch(NSException *e)
        {
            NSLog(@"%@", e.reason);
            ret = 1;
        }

        [PTPDFNet Terminate: 0];
        return ret;
    }
}

