//---------------------------------------------------------------------------------------
// Copyright (c) 2001-2025 by Apryse Software Inc. All Rights Reserved.
// Consult legal.txt regarding legal and license information.
//---------------------------------------------------------------------------------------

#import <OBJC/PDFNetOBJC.h>
#import <Foundation/Foundation.h>

//---------------------------------------------------------------------------------------
// The following sample illustrates how to use the PDF::Convert utility class to convert 
// documents and files to HTML.
//
// There are two HTML modules and one of them is an optional PDFNet Add-on.
// 1. The built-in HTML module is used to convert PDF documents to fixed-position HTML
//    documents.
// 2. The optional Structured Output add-on module is used to convert PDF documents to
//    HTML documents with text flowing across the browser window.
//
// The Apryse SDK Structured Output module can be downloaded from
// https://docs.apryse.com/documentation/core/info/modules/
//
// Please contact us if you have any questions.
//---------------------------------------------------------------------------------------

int main (int argc, const char * argv[])
{
    @autoreleasepool {

        NSString *inputPath = @"../../TestFiles/";
        NSString *outputPath = @"../../TestFiles/Output/";

        // The first step in every application using PDFNet is to initialize the 
        // library. The library is usually initialized only once, but calling 
        // Initialize() multiple times is also fine.
        [PTPDFNet Initialize: 0];

        int ret = 0;

        //------------------------------------------------------------------------

        @try {
            // Convert PDF document to HTML with fixed positioning option turned on (default)
            NSLog(@"Converting PDF to HTML with fixed positioning option turned on (default)");

            NSString *inputFile = [inputPath stringByAppendingString:@"paragraphs_and_tables.pdf"];
            NSString *outputFile = [outputPath stringByAppendingString:@"paragraphs_and_tables_fixed_positioning.html"];

            // Convert PDF to HTML
            [PTConvert ToHtml:inputFile out_path:outputFile];

            NSLog(@"Result saved in %@", outputFile);
        }
        @catch (NSException *e) {
            NSLog(@"Exception: %@ - %@\n", e.name, e.reason);
            ret = 1;
        }

        //------------------------------------------------------------------------

        [PTPDFNet AddResourceSearchPath:@"../../../Lib/"];

        if (![PTStructuredOutputModule IsModuleAvailable]) {
            NSLog(@"");
            NSLog(@"Unable to run the sample: Apryse SDK Structured Output module not available.");
            NSLog(@"---------------------------------------------------------------");
            NSLog(@"The Structured Output module is an optional add-on, available for download");
            NSLog(@"at https://docs.apryse.com/documentation/core/info/modules/. If you have already");
            NSLog(@"downloaded this module, ensure that the SDK is able to find the required files");
            NSLog(@"using the PDFNet::AddResourceSearchPath() function.");
            NSLog(@"");

            return 1;
        }
        
        //------------------------------------------------------------------------

        @try {
            // Convert PDF document to HTML with reflow full option turned on (1)
            NSLog(@"Converting PDF to HTML with reflow full option turned on (1)");

            NSString *inputFile = [inputPath stringByAppendingString:@"paragraphs_and_tables.pdf"];
            NSString *outputFile = [outputPath stringByAppendingString:@"paragraphs_and_tables_reflow_full.html"];

            PTHTMLOutputOptions *htmlOutputOptions = [[PTHTMLOutputOptions alloc] init];

            // Set e_reflow_full content reflow setting
            [htmlOutputOptions SetContentReflowSetting:e_pthtml_reflow_full];

            // Convert PDF to HTML
            [PTConvert ToHtmlWithFilename:inputFile out_path:outputFile options:htmlOutputOptions];

            NSLog(@"Result saved in %@", outputFile);
        }
        @catch (NSException *e) {
            NSLog(@"Exception: %@ - %@\n", e.name, e.reason);
            ret = 1;
        }

        //------------------------------------------------------------------------

        @try {
            // Convert PDF document to HTML with reflow full option turned on (only converting the first page) (2)
            NSLog(@"Converting PDF to HTML with reflow full option turned on (only converting the first page) (2)");

            NSString *inputFile = [inputPath stringByAppendingString:@"paragraphs_and_tables.pdf"];
            NSString *outputFile = [outputPath stringByAppendingString:@"paragraphs_and_tables_reflow_full_first_page.html"];

            PTHTMLOutputOptions *htmlOutputOptions = [[PTHTMLOutputOptions alloc] init];

            // Set e_reflow_full content reflow setting
            [htmlOutputOptions SetContentReflowSetting:e_pthtml_reflow_full];

            // Convert only the first page
            [htmlOutputOptions SetPages:1 page_to:1];

            // Convert PDF to HTML
            [PTConvert ToHtmlWithFilename:inputFile out_path:outputFile options:htmlOutputOptions];

            NSLog(@"Result saved in %@", outputFile);
        }
        @catch (NSException *e) {
            NSLog(@"Exception: %@ - %@\n", e.name, e.reason);
            ret = 1;
        }

        //------------------------------------------------------------------------

        [PTPDFNet Terminate: 0];
        return ret;
    }
}
