//---------------------------------------------------------------------------------------
// Copyright (c) 2001-2025 by Apryse Software Inc. All Rights Reserved.
// Consult legal.txt regarding legal and license information.
//---------------------------------------------------------------------------------------

#import <OBJC/PDFNetOBJC.h>
#import <Foundation/Foundation.h>

//---------------------------------------------------------------------------------------
// The following sample illustrates how to use the PDF::Convert utility class to convert 
// documents and files to Word, Excel and PowerPoint.
//
// The Structured Output module is an optional PDFNet Add-on that can be used to convert PDF
// and other documents into Word, Excel, PowerPoint and HTML format.
//
// The Apryse SDK Structured Output module can be downloaded from
// https://docs.apryse.com/documentation/core/info/modules/
//
// Please contact us if you have any questions.
//---------------------------------------------------------------------------------------

int main (int argc, const char * argv[])
{
    @autoreleasepool {

        NSString *inputPath = @"../../TestFiles/";
        NSString *outputPath = @"../../TestFiles/Output/";

        // The first step in every application using PDFNet is to initialize the 
        // library. The library is usually initialized only once, but calling 
        // Initialize() multiple times is also fine.
        [PTPDFNet Initialize: 0];

        int ret = 0;

        //------------------------------------------------------------------------

        [PTPDFNet AddResourceSearchPath:@"../../../Lib/"];

        if (![PTStructuredOutputModule IsModuleAvailable]) {
            NSLog(@"");
            NSLog(@"Unable to run the sample: Apryse SDK Structured Output module not available.");
            NSLog(@"---------------------------------------------------------------");
            NSLog(@"The Structured Output module is an optional add-on, available for download");
            NSLog(@"at https://docs.apryse.com/documentation/core/info/modules/. If you have already");
            NSLog(@"downloaded this module, ensure that the SDK is able to find the required files");
            NSLog(@"using the PDFNet::AddResourceSearchPath() function.");
            NSLog(@"");

            return 1;
        }
        
        //------------------------------------------------------------------------
        // Word
        //------------------------------------------------------------------------

        @try {
            // Convert PDF document to Word
            NSLog(@"Converting PDF to Word");

            NSString *inputFile = [inputPath stringByAppendingString:@"paragraphs_and_tables.pdf"];
            NSString *outputFile = [outputPath stringByAppendingString:@"paragraphs_and_tables.docx"];

            // Convert PDF to Word
            [PTConvert ToWord:inputFile out_path:outputFile];

            NSLog(@"Result saved in %@", outputFile);
        }
        @catch (NSException *e) {
            NSLog(@"Exception: %@ - %@\n", e.name, e.reason);
            ret = 1;
        }

        @try {
            // Convert PDF document to Word with options
            NSLog(@"Converting PDF to Word with options");

            NSString *inputFile = [inputPath stringByAppendingString:@"paragraphs_and_tables.pdf"];
            NSString *outputFile = [outputPath stringByAppendingString:@"paragraphs_and_tables_first_page.docx"];

            PTWordOutputOptions *wordOutputOptions = [[PTWordOutputOptions alloc] init];

            // Convert only the first page
            [wordOutputOptions SetPages:1 page_to:1];

            // Convert PDF to Word
            [PTConvert ToWordWithFilename:inputFile out_path:outputFile options:wordOutputOptions];

            NSLog(@"Result saved in %@", outputFile);
        }
        @catch (NSException *e) {
            NSLog(@"Exception: %@ - %@\n", e.name, e.reason);
            ret = 1;
        }

        //------------------------------------------------------------------------
        // Excel
        //------------------------------------------------------------------------

        @try {
            // Convert PDF document to Excel
            NSLog(@"Converting PDF to Excel");

            NSString *inputFile = [inputPath stringByAppendingString:@"paragraphs_and_tables.pdf"];
            NSString *outputFile = [outputPath stringByAppendingString:@"paragraphs_and_tables.xlsx"];

            // Convert PDF to Excel
            [PTConvert ToExcel:inputFile out_path:outputFile];

            NSLog(@"Result saved in %@", outputFile);
        }
        @catch (NSException *e) {
            NSLog(@"Exception: %@ - %@\n", e.name, e.reason);
            ret = 1;
        }

        @try {
            // Convert PDF document to Excel with options
            NSLog(@"Converting PDF to Excel with options");

            NSString *inputFile = [inputPath stringByAppendingString:@"paragraphs_and_tables.pdf"];
            NSString *outputFile = [outputPath stringByAppendingString:@"paragraphs_and_tables_second_page.xlsx"];

            PTExcelOutputOptions *excelOutputOptions = [[PTExcelOutputOptions alloc] init];

            // Convert only the second page
            [excelOutputOptions SetPages:2 page_to:2];

            // Convert PDF to Excel
            [PTConvert ToExcelWithFilename:inputFile out_path:outputFile options:excelOutputOptions];

            NSLog(@"Result saved in %@", outputFile);
        }
        @catch (NSException *e) {
            NSLog(@"Exception: %@ - %@\n", e.name, e.reason);
            ret = 1;
        }

        //------------------------------------------------------------------------
        // PowerPoint
        //------------------------------------------------------------------------

        @try {
            // Convert PDF document to PowerPoint
            NSLog(@"Converting PDF to PowerPoint");

            NSString *inputFile = [inputPath stringByAppendingString:@"paragraphs_and_tables.pdf"];
            NSString *outputFile = [outputPath stringByAppendingString:@"paragraphs_and_tables.pptx"];

            // Convert PDF to PowerPoint
            [PTConvert ToPowerPoint:inputFile out_path:outputFile];

            NSLog(@"Result saved in %@", outputFile);
        }
        @catch (NSException *e) {
            NSLog(@"Exception: %@ - %@\n", e.name, e.reason);
            ret = 1;
        }

        @try {
            // Convert PDF document to PowerPoint with options
            NSLog(@"Converting PDF to PowerPoint with options");

            NSString *inputFile = [inputPath stringByAppendingString:@"paragraphs_and_tables.pdf"];
            NSString *outputFile = [outputPath stringByAppendingString:@"paragraphs_and_tables_first_page.pptx"];

            PTPowerPointOutputOptions *powerPointOutputOptions = [[PTPowerPointOutputOptions alloc] init];

            // Convert only the first page
            [powerPointOutputOptions SetPages:1 page_to:1];

            // Convert PDF to PowerPoint
            [PTConvert ToPowerPointWithFilename:inputFile out_path:outputFile options:powerPointOutputOptions];

            NSLog(@"Result saved in %@", outputFile);
        }
        @catch (NSException *e) {
            NSLog(@"Exception: %@ - %@\n", e.name, e.reason);
            ret = 1;
        }

        //------------------------------------------------------------------------

        [PTPDFNet Terminate: 0];
        return ret;
    }
}
