//---------------------------------------------------------------------------------------
// Copyright (c) 2001-2025 by Apryse Software Inc. All Rights Reserved.
// Consult legal.txt regarding legal and license information.
//---------------------------------------------------------------------------------------

#import <OBJC/PDFNetOBJC.h>
#import <Foundation/Foundation.h>
#include <math.h>

//---------------------------------------------------------------------------------------
// The following sample illustrates how to use the UndoRedo API.
//---------------------------------------------------------------------------------------
int main(int argc, char *argv[])
{
    @autoreleasepool
    {
        int ret = 0;
        
        @try
        {
            // The first step in every application using PDFNet is to initialize the 
            // library and set the path to common PDF resources. The library is usually 
            // initialized only once, but calling Initialize() multiple times is also fine.
            [PTPDFNet Initialize: 0];

            // Open the PDF document.
            PTPDFDoc *doc = [[PTPDFDoc alloc] initWithFilepath: @"../../TestFiles/newsletter.pdf"];
            
            PTSDFUndoManager* undo_manager = [doc GetUndoManager];
            
            // Take a snapshot to which we can undo after making changes.
            PTSDFResultSnapshot* snap0 = [undo_manager TakeSnapshot];
            
            PTSDFDocSnapshot* snap0_state = [snap0 CurrentState];
            
            PTElementBuilder* builder = [[PTElementBuilder alloc] init];    // Used to build new Element objects
            PTElementWriter* writer = [[PTElementWriter alloc] init];       // Used to write Elements to the page  
            
            PTPDFRect * rect = [[PTPDFRect alloc] init]; 
            [rect Set: 0 y1: 0 x2: 612 y2: 792];
            
            PTPage* page = [doc PageCreate: rect];  // Start a new page
            
            [writer WriterBeginWithPage: page placement: e_ptoverlay page_coord_sys: YES compress: YES resources: NULL];    // Begin writing to this page

            // ----------------------------------------------------------
            // Add JPEG image to the file
            PTImage* img = [PTImage Create: [doc GetSDFDoc] filename: @"../../TestFiles/peppers.jpg"];
            PTElement* element = [builder CreateImageWithMatrix: img mtx: [[PTMatrix2D alloc] initWithA: 200 b: 0 c: 0 d: 250 h: 50 v: 500]];
            [writer WritePlacedElement: element];
            
            [writer End];    // Finish writing to the page
            [doc PagePushFront: page];
            
            // Take a snapshot after making changes, so that we can redo later (after undoing first).
            PTSDFResultSnapshot* snap1 = [undo_manager TakeSnapshot];
            
            if ([[snap1 PreviousState] Equals: snap0_state])
            {
                NSLog(@"snap1 previous state equals snap0_state; previous state is correct");
            }
            
            PTSDFDocSnapshot* snap1_state = [snap1 CurrentState];
            
            [doc SaveToFile: @"../../TestFiles/Output/addimage.pdf" flags: e_ptincremental];
            
            if ([undo_manager CanUndo])
            {
                PTSDFResultSnapshot* undo_snap = [undo_manager Undo];
                
                [doc SaveToFile: @"../../TestFiles/Output/addimage_undone.pdf" flags: e_ptincremental];
                
                PTSDFDocSnapshot* undo_snap_state = [undo_snap CurrentState];
                
                if ([undo_snap_state Equals: snap0_state])
                {
                    NSLog(@"undo_snap_state equals snap0_state; undo was successful");
                }
                
                if ([undo_manager CanRedo])
                {
                    PTSDFResultSnapshot* redo_snap = [undo_manager Redo];
                    
                    [doc SaveToFile: @"../../TestFiles/Output/addimage_redone.pdf" flags: e_ptincremental];
                    
                    if ([[redo_snap PreviousState] Equals: undo_snap_state])
                    {
                        NSLog(@"redo_snap previous state equals undo_snap_state; previous state is correct");
                    }
                    
                    PTSDFDocSnapshot* redo_snap_state = [redo_snap CurrentState];
                    
                    if ([redo_snap_state Equals: snap1_state])
                    {
                        NSLog(@"Snap1 and redo_snap are equal; redo was successful");
                    }
                }
                else
                {
                    NSLog(@"Problem encountered - cannot redo.");
                    ret = 1;
                }
            }
            else
            {
                NSLog(@"Problem encountered - cannot undo.");
                ret = 1;
            }
        }
        @catch(NSException *e)    
        {
            NSLog(@"%@", e.reason);
            ret = 1;
        }
        [PTPDFNet Terminate: 0];
        return ret;
    }
}

