'---------------------------------------------------------------------------------------
' Copyright (c) 2001-2025 by Apryse Software Inc. All Rights Reserved.
' Consult legal.txt regarding legal and license information.
'---------------------------------------------------------------------------------------
Imports System

Imports pdftron
Imports pdftron.Common
Imports pdftron.PDF

' <summary>
'---------------------------------------------------------------------------------------
' The Barcode Module is an optional PDFNet add-on that can be used to extract
' various types of barcodes from PDF documents.
'
' The Apryse SDK Barcode Module can be downloaded from https://dev.apryse.com/
'---------------------------------------------------------------------------------------
' </summary>
Module BarcodeTestVB
    Dim pdfNetLoader As PDFNetLoader
    Sub New()
        pdfNetLoader = pdftron.PDFNetLoader.Instance()
    End Sub

    ' The main entry point for the application.
    Sub Main()

        ' The first step in every application using PDFNet is to initialize the 
        ' library and set the path to common PDF resources. The library is usually 
        ' initialized only once, but calling Initialize() multiple times is also fine.
        PDFNet.Initialize(PDFTronLicense.Key)

        ' Can optionally set path to the Barcode module
        PDFNet.AddResourceSearchPath("../../../../../Lib/")

        ' Test if the add-on is installed
        If Not BarcodeModule.IsModuleAvailable() Then
            Console.WriteLine("")
            Console.WriteLine("Unable to run BarcodeTest: Apryse SDK Barcode Module not available.")
            Console.WriteLine("---------------------------------------------------------------")
            Console.WriteLine("The Barcode Module is an optional add-on, available for download")
            Console.WriteLine("at https://dev.apryse.com/. If you have already downloaded this")
            Console.WriteLine("module, ensure that the SDK is able to find the required files")
            Console.WriteLine("using the PDFNet.AddResourceSearchPath() function.")
            Console.WriteLine("")
            Return
        End If

        ' Relative path to the folder containing test files.
        Dim input_path As String = "../../../../TestFiles/Barcode/"
        Dim output_path As String = "../../../../TestFiles/Output/"

        '--------------------------------------------------------------------------------
        ' Example 1) Detect and extract all barcodes from a PDF document into a JSON file
        Try
            Console.WriteLine("Example 1: extracting barcodes from barcodes.pdf to barcodes.json")

            ' A) Open the .pdf document
            Using doc As PDFDoc = New PDFDoc(input_path & "barcodes.pdf")

                ' B) Detect PDF barcodes with the default options
                BarcodeModule.ExtractBarcodes(doc, output_path & "barcodes.json")

            End Using
        Catch e As PDFNetException
            Console.WriteLine(e.Message)
        End Try

        '--------------------------------------------------------------------------------
        ' Example 2) Limit barcode extraction to a range of pages, and retrieve the JSON into a
        ' local string variable, which is then written to a file in a separate function call
        Try
            Console.WriteLine("Example 2: extracting barcodes from pages 1-2 to barcodes_from_pages_1-2.json")

            ' A) Open the .pdf document
            Using doc As PDFDoc = New PDFDoc(input_path & "barcodes.pdf")

                ' B) Detect PDF barcodes with custom options
                Dim options As BarcodeOptions = New BarcodeOptions()

                ' Convert only the first two pages
                options.SetPages("1-2")

                Dim json As String = BarcodeModule.ExtractBarcodesAsString(doc, options)

                ' C) Save JSON to file
                System.IO.File.WriteAllText(output_path & "barcodes_from_pages_1-2.json", json)

            End Using
        Catch e As PDFNetException
            Console.WriteLine(e.Message)
        End Try

        '--------------------------------------------------------------------------------
        ' Example 3) Narrow down barcode types and allow the detection of both horizontal
        ' and vertical barcodes
        Try
            Console.WriteLine("Example 3: extracting basic horizontal and vertical barcodes")

            ' A) Open the .pdf document
            Using doc As PDFDoc = New PDFDoc(input_path & "barcodes.pdf")

                ' B) Detect only basic 1D barcodes, both horizontal and vertical
                Dim options As BarcodeOptions = New BarcodeOptions()

                ' Limit extraction to basic 1D barcode types, such as EAN 13, EAN 8, UPCA, UPCE,
                ' Code 3 of 9, Code 128, Code 2 of 5, Code 93, Code 11 and GS1 Databar.
                options.SetBarcodeSearchTypes(BarcodeOptions.BarcodeTypeGroup.e_linear)

                ' Search for barcodes oriented horizontally and vertically
                options.SetBarcodeOrientations(BarcodeOptions.BarcodeOrientation.e_horizontal Or BarcodeOptions.BarcodeOrientation.e_vertical)

                BarcodeModule.ExtractBarcodes(doc, output_path & "barcodes_1D.json", options)

            End Using
        Catch e As PDFNetException
            Console.WriteLine(e.Message)
        End Try

        Console.WriteLine("Done.")
        PDFNet.Terminate()
    End Sub

End Module
