'
' Copyright (c) 2001-2025 by Apryse Software Inc. All Rights Reserved.
'

Imports System

Imports pdftron
Imports pdftron.Common
Imports pdftron.Filters
Imports pdftron.SDF
Imports pdftron.PDF

Module BookmarkTestVB2003
    Dim pdfNetLoader As PDFNetLoader
    Sub New()
        pdfNetLoader = pdftron.PDFNetLoader.Instance()
    End Sub
    '-----------------------------------------------------------------------------------------
    ' The sample code illustrates how to read, write, and edit existing outline items 
    ' and create new bookmarks using both the high-level and the SDF/Cos API.
    '-----------------------------------------------------------------------------------------
    Sub Main()
        BookmarkTest.RunTest()
    End Sub


    Class BookmarkTest

        Public Shared Sub PrintIndent(ByVal item As Bookmark)
            Dim indent As Integer = item.GetIndent() - 1
            Dim i As Integer
            For i = 1 To indent
                Console.Write("  ")
            Next
        End Sub

        Public Shared Sub PrintOutlineTree(ByVal item As Bookmark)

            Do While item.IsValid()

                PrintIndent(item)
                If item.IsOpen Then
                    Console.Write("- {0:s} ACTION -> ", item.GetTitle())
                Else
                    Console.Write("+ {0:s} ACTION -> ", item.GetTitle())
                End If


                ' Print Action
                Dim action As pdftron.PDF.Action = item.GetAction()
                If action.IsValid() Then
                    If action.GetType() = pdftron.PDF.Action.Type.e_GoTo Then
                        Dim dest As Destination = action.GetDest()
                        If (dest.IsValid()) Then
                            Dim page As Page = dest.GetPage()
                            Console.WriteLine("GoTo Page #{0:d}", page.GetIndex())
                        End If
                    Else
                        Console.WriteLine("Not a 'GoTo' action")
                    End If
                Else
                    Console.WriteLine("NULL")
                End If

                If item.HasChildren() Then   ' Recursively print children sub-trees
                    PrintOutlineTree(item.GetFirstChild())
                End If
                item = item.GetNext()
            Loop
        End Sub


        Shared Sub RunTest()

            PDFNet.Initialize(PDFTronLicense.Key)

            ' Relative path to the folder containing test files.
            Dim input_path As String = "../../../../TestFiles/"
            Dim output_path As String = "../../../../TestFiles/Output/"


            ' The following example illustrates how to create and edit the outline tree 
            ' using high-level Bookmark methods.
            Try
                Using doc1 As PDFDoc = New PDFDoc(input_path + "numbered.pdf")
                    doc1.InitSecurityHandler()


                    ' Lets first create the root bookmark items. 
                    Dim red As Bookmark = Bookmark.Create(doc1, "Red")
                    Dim green As Bookmark = Bookmark.Create(doc1, "Green")
                    Dim blue As Bookmark = Bookmark.Create(doc1, "Blue")

                    doc1.AddRootBookmark(red)
                    doc1.AddRootBookmark(green)
                    doc1.AddRootBookmark(blue)

                    ' You can also add new root bookmarks using Bookmark.AddNext("...")
                    blue.AddNext("foo")
                    blue.AddNext("bar")

                    ' We can now associate new bookmarks with page destinations:

                    ' The following example creates an 'explicit' destination (see 
                    ' section '8.2.1 Destinations' in PDF Reference for more details)
                    Dim red_dest As Destination = Destination.CreateFit(doc1.GetPage(1))
                    red.SetAction(pdftron.PDF.Action.CreateGoto(red_dest))

                    ' Create an explicit destination to the first green page in the document
                    green.SetAction(pdftron.PDF.Action.CreateGoto(Destination.CreateFit(doc1.GetPage(10))))

                    ' The following example creates a 'named' destination (see 
                    ' section '8.2.1 Destinations' in PDF Reference for more details)
                    ' Named destinations have certain advantages over explicit destinations.
                    Dim key As String = "blue1"
                    Dim blue_action As pdftron.PDF.Action = pdftron.PDF.Action.CreateGoto(key, Destination.CreateFit(doc1.GetPage(19)))

                    blue.SetAction(blue_action)

                    ' We can now add children Bookmarks
                    Dim sub_red1 As Bookmark = red.AddChild("Red - Page 1")
                    sub_red1.SetAction(pdftron.PDF.Action.CreateGoto(Destination.CreateFit(doc1.GetPage(1))))
                    Dim sub_red2 As Bookmark = red.AddChild("Red - Page 2")
                    sub_red2.SetAction(pdftron.PDF.Action.CreateGoto(Destination.CreateFit(doc1.GetPage(2))))
                    Dim sub_red3 As Bookmark = red.AddChild("Red - Page 3")
                    sub_red3.SetAction(pdftron.PDF.Action.CreateGoto(Destination.CreateFit(doc1.GetPage(3))))
                    Dim sub_red4 As Bookmark = sub_red3.AddChild("Red - Page 4")
                    sub_red4.SetAction(pdftron.PDF.Action.CreateGoto(Destination.CreateFit(doc1.GetPage(4))))
                    Dim sub_red5 As Bookmark = sub_red3.AddChild("Red - Page 5")
                    sub_red5.SetAction(pdftron.PDF.Action.CreateGoto(Destination.CreateFit(doc1.GetPage(5))))
                    Dim sub_red6 As Bookmark = sub_red3.AddChild("Red - Page 6")
                    sub_red6.SetAction(pdftron.PDF.Action.CreateGoto(Destination.CreateFit(doc1.GetPage(6))))

                    ' Example of how to find and delete a bookmark by title text.
                    Dim foo As Bookmark = doc1.GetFirstBookmark().Find("foo")
                    If foo.IsValid() Then
                        foo.Delete()
                    End If
                    Dim bar As Bookmark = doc1.GetFirstBookmark().Find("bar")
                    If bar.IsValid() Then
                        bar.Delete()
                    End If

                    ' Adding color to Bookmarks. Color and other formatting can help readers 
                    ' get around more easily in large PDF documents.
                    red.SetColor(1, 0, 0)
                    green.SetColor(0, 1, 0)
                    green.SetFlags(2)     ' set bold font
                    blue.SetColor(0, 0, 1)
                    blue.SetFlags(3)   ' set bold and italic

                    doc1.Save(output_path + "bookmark.pdf", 0)
                End Using
                Console.WriteLine("Done. Result saved in bookmark.pdf")
            Catch e As PDFNetException
                Console.WriteLine(e.Message)
            End Try


            ' The following example illustrates how to traverse the outline tree using 
            ' Bookmark navigation methods: Bookmark.GetNext(), Bookmark.GetPrev(), 
            ' Bookmark.GetFirstChild () and Bookmark.GetLastChild ().
            Try
                ' Open the document that was saved in the previous code sample
                Using doc1 As PDFDoc = New PDFDoc(output_path + "bookmark.pdf")
                    doc1.InitSecurityHandler()

                    Dim root As Bookmark = doc1.GetFirstBookmark()
                    PrintOutlineTree(root)
                    Console.WriteLine("Done.")
                End Using
            Catch e As PDFNetException
                Console.WriteLine(e.Message)
            End Try

            ' The following example illustrates how to create a Bookmark to a page 
            ' in a remote document. A remote go-to action is similar to an ordinary 
            ' go-to action, but jumps to a destination in another PDF file instead 
            ' of the current file. See Section 8.5.3 'Remote Go-To Actions' in PDF 
            ' Reference Manual for details.
            Try
                Using doc1 As PDFDoc = New PDFDoc(output_path + "bookmark.pdf")
                    doc1.InitSecurityHandler()

                    ' Create file specification (the file referred to by the remote bookmark)
                    Dim file_spec As Obj = doc1.CreateIndirectDict()
                    file_spec.PutName("Type", "Filespec")
                    file_spec.PutString("F", "bookmark.pdf")

                    Dim spec As FileSpec = New FileSpec(file_spec)
                    Dim goto_remote As pdftron.PDF.Action = pdftron.PDF.Action.CreateGotoRemote(spec, 5, True)

                    Dim remoteBookmark1 As Bookmark = Bookmark.Create(doc1, "REMOTE BOOKMARK 1")
                    remoteBookmark1.SetAction(goto_remote)
                    doc1.AddRootBookmark(remoteBookmark1)

                    ' Create another remote bookmark, but this time using the low-level SDF/Cos API.
                    Dim remoteBookmark2 As Bookmark = Bookmark.Create(doc1, "REMOTE BOOKMARK 2")
                    doc1.AddRootBookmark(remoteBookmark2)
                    Dim gotoR As Obj = remoteBookmark2.GetSDFObj().PutDict("A")

                    gotoR.PutName("S", "GoToR")  ' Set action type
                    gotoR.PutBool("NewWindow", True)

                    ' Set the file specification
                    gotoR.Put("F", file_spec)

                    ' Set the destination
                    Dim dest As Obj = gotoR.PutArray("D")
                    dest.PushBackNumber(9) ' Jump to the tenth page. Note that Acrobat indexes pages from 0.
                    dest.PushBackName("Fit") ' Fit the page

                    doc1.Save(output_path + "bookmark_remote.pdf", SDFDoc.SaveOptions.e_linearized)
                End Using
                Console.WriteLine("Done. Result saved in bookmark_remote.pdf")
            Catch e As PDFNetException
                Console.WriteLine(e.Message)
            End Try
            PDFNet.Terminate()
        End Sub

    End Class

End Module
