'
' Copyright (c) 2001-2025 by Apryse Software Inc. All Rights Reserved.
'

Imports System
Imports System.Drawing
Imports System.Drawing.Drawing2D
Imports pdftron
Imports pdftron.Common
Imports pdftron.Filters
Imports pdftron.SDF
Imports pdftron.PDF

' The following sample illustrates how to convert to PDF with virtual printer on Windows.
' It supports several input formats like docx, xlsx, rtf, txt, html, pub, emf, etc. For more details, visit 
' https://docs.apryse.com/documentation/windows/guides/features/conversion/convert-other/
' 
' To check if ToPDF (or ToXPS) require that PDFNet printer is installed use Convert::RequiresPrinter(filename). 
' The installing application must be run as administrator. The manifest for this sample 
' specifies appropriate the UAC elevation.
' 
' Note: the PDFNet printer is a virtual XPS printer supported on Vista SP1 and Windows 7, or .NET Framework
' 3.x or higher. For older versions of .NET Framework running on Windows XP or Vista SP0 you need to install 
' the XPS Essentials Pack (or equivalent redistributables). 
' 
' Also note that conversion under ASP.NET can be tricky to configure. Please see the following document for advice: 
' http://www.pdftron.com/pdfnet/faq_files/Converting_Documents_in_Windows_Service_or_ASP.NET_Application_using_PDFNet.pdf
Module ConvertPrintTestVB
    Class Testfile
        Public inputFile, outputFile As String

        Public Sub New(ByVal inFile As String, ByVal outFile As String)
            inputFile = inFile
            outputFile = outFile
        End Sub
    End Class

    Class Class1
        Shared pdfNetLoader As pdftron.PDFNetLoader = pdftron.PDFNetLoader.Instance()

        Shared Sub New()
        End Sub

        Const inputPath As String = "../../../../TestFiles/"
        Const outputPath As String = "../../../../TestFiles/Output/"

        Private Shared Function ConvertSpecificFormats() As Boolean
            Dim err As Boolean = False
            Try
                Using pdfdoc As PDFDoc = New PDFDoc()
                    Console.WriteLine("Converting from EMF")
                    pdftron.PDF.Convert.FromEmf(pdfdoc, inputPath & "simple-emf.emf")
                    pdfdoc.Save(outputPath & "emf2pdf v2.pdf", SDFDoc.SaveOptions.e_remove_unused)
                    Console.WriteLine("Saved emf2pdf v2.pdf")
                End Using

            Catch e As PDFNetException
                Console.WriteLine(e.Message)
                err = True
            End Try

            Try
                Console.WriteLine("Converting DOCX to XPS")
                pdftron.PDF.Convert.ToXps(inputPath & "simple-word_2007.docx", outputPath & "simple-word_2007.xps")
                Console.WriteLine("Saved simple-word_2007.xps")
            Catch e As PDFNetException
                Console.WriteLine(e.Message)
                err = True
            End Try

            Return err
        End Function

        Private Shared Function ConvertToPdfFromFile() As Boolean
            Dim testfiles As System.Collections.ArrayList = New System.Collections.ArrayList()
            testfiles.Add(New ConvertPrintTestVB.Testfile("simple-powerpoint_2007.pptx", "pptx2pdf.pdf"))
            testfiles.Add(New ConvertPrintTestVB.Testfile("simple-text.txt", "txt2pdf.pdf"))
            testfiles.Add(New ConvertPrintTestVB.Testfile("simple-word_2007.docx", "docx2pdf.pdf"))
            testfiles.Add(New ConvertPrintTestVB.Testfile("simple-rtf.rtf", "rtf2pdf.pdf"))
            testfiles.Add(New ConvertPrintTestVB.Testfile("simple-excel_2007.xlsx", "xlsx2pdf.pdf"))
            testfiles.Add(New ConvertPrintTestVB.Testfile("simple-publisher.pub", "pub2pdf.pdf"))
            testfiles.Add(New ConvertPrintTestVB.Testfile("simple-emf.emf", "emf2pdf.pdf"))
            testfiles.Add(New ConvertPrintTestVB.Testfile("simple-webpage.mht", "mht2pdf.pdf"))
            testfiles.Add(New ConvertPrintTestVB.Testfile("simple-webpage.html", "html2pdf.pdf"))
            Dim err As Boolean = False

            Try
                If pdftron.PDF.Convert.Printer.IsInstalled("PDFTron PDFNet") Then
                    pdftron.PDF.Convert.Printer.SetPrinterName("PDFTron PDFNet")
                ElseIf Not pdftron.PDF.Convert.Printer.IsInstalled() Then

                    Try
                        Console.WriteLine("Installing printer (requires Windows platform and administrator)")
                        pdftron.PDF.Convert.Printer.Install()
                        Console.WriteLine("Installed printer " & pdftron.PDF.Convert.Printer.GetPrinterName())
                    Catch e As PDFNetException
                        Console.WriteLine("ERROR: Unable to install printer.")
                        Console.WriteLine(e.Message)
                        err = True
                    Catch
                        Console.WriteLine("ERROR: Unable to install printer. Make sure that the package's bitness matches your operating system's bitness and that you are running with administrator privileges.")
                    End Try
                End If

            Catch e As PDFNetException
                Console.WriteLine("ERROR: Unable to install printer.")
                Console.WriteLine(e.Message)
                err = True
            End Try

            For Each file As Testfile In testfiles

                Try

                    Using pdfdoc As pdftron.PDF.PDFDoc = New PDFDoc()

                        If pdftron.PDF.Convert.RequiresPrinter(inputPath & file.inputFile) Then
                            Console.WriteLine("Using PDFNet printer to convert file " & file.inputFile)
                        End If

                        pdftron.PDF.Convert.ToPdf(pdfdoc, inputPath & file.inputFile)
                        pdfdoc.Save(outputPath & file.outputFile, SDFDoc.SaveOptions.e_linearized)
                        Console.WriteLine("Converted file: " & file.inputFile)
                        Console.WriteLine("to: " & file.outputFile)
                    End Using

                Catch e As PDFNetException
                    Console.WriteLine("ERROR: on input file " & file.inputFile)
                    Console.WriteLine(e.Message)
                    err = True
                End Try
            Next

            Return err
        End Function

        <STAThread>
        Shared Sub Main(ByVal args As String())
            ' Virtual printer only available on Windows
            If Environment.OSVersion.Platform = PlatformID.Win32NT Then
                PDFNet.Initialize(PDFTronLicense.Key)
                Dim err As Boolean = False
                err = ConvertToPdfFromFile()

                If err Then
                    Console.WriteLine("ConvertFile failed")
                Else
                    Console.WriteLine("ConvertFile succeeded")
                End If

                err = ConvertSpecificFormats()

                If err Then
                    Console.WriteLine("ConvertSpecificFormats failed")
                Else
                    Console.WriteLine("ConvertSpecificFormats succeeded")
                End If

                If pdftron.PDF.Convert.Printer.IsInstalled() Then
                    Try
                        Console.WriteLine("Uninstalling printer (requires Windows platform and administrator)")
                        pdftron.PDF.Convert.Printer.Uninstall()
                        Console.WriteLine("Uninstalled Printer " & pdftron.PDF.Convert.Printer.GetPrinterName())
                    Catch
                        Console.WriteLine("Unable to uninstall printer")
                    End Try
                End If

                PDFNet.Terminate()
                Console.WriteLine("Done.")
            Else
                Console.WriteLine("ConvertPrintTest only available on Windows")
            End If
        End Sub
    End Class
End Module
