'
' Copyright (c) 2001-2025 by Apryse Software Inc. All Rights Reserved.
'

Imports System
Imports System.Drawing
Imports System.Drawing.Drawing2D
Imports pdftron
Imports pdftron.Common
Imports pdftron.Filters
Imports pdftron.SDF
Imports pdftron.PDF

' The following sample illustrates how to use the PDF::Convert utility class to convert 
' documents and files to PDF, XPS, or SVG, or EMF. The sample also shows how to convert MS Office files 
' using our built in conversion.
' 
' Certain file formats such as XPS, EMF, PDF, and raster image formats can be directly 
' converted to PDF or XPS. 
' 
' Also note that conversion under ASP.NET can be tricky to configure. Please see the following document for advice: 
' http://www.pdftron.com/pdfnet/faq_files/Converting_Documents_in_Windows_Service_or_ASP.NET_Application_using_PDFNet.pdf
Module ConvertTestVB
    Class Testfile
        Public inputFile, outputFile As String

        Public Sub New(ByVal inFile As String, ByVal outFile As String)
            inputFile = inFile
            outputFile = outFile
        End Sub
    End Class

    Class Class1
        Shared pdfNetLoader As pdftron.PDFNetLoader = pdftron.PDFNetLoader.Instance()

        Shared Sub New()
        End Sub

        Const inputPath As String = "../../../../TestFiles/"
        Const outputPath As String = "../../../../TestFiles/Output/"

        Private Shared Function ConvertSpecificFormats() As Boolean
            Dim err As Boolean = False

            Try

                Using pdfdoc As PDFDoc = New PDFDoc()
                    Console.WriteLine("Converting from XPS")
                    pdftron.PDF.Convert.FromXps(pdfdoc, inputPath & "simple-xps.xps")
                    pdfdoc.Save(outputPath & "xps2pdf v2.pdf", SDFDoc.SaveOptions.e_remove_unused)
                    Console.WriteLine("Saved xps2pdf v2.pdf")
                End Using

            Catch e As PDFNetException
                Console.WriteLine(e.Message)
                err = True
            End Try

            Try
                Using pdfdoc As PDFDoc = New PDFDoc()
                    Dim [set] As ObjSet = New ObjSet()
                    Dim options As Obj = [set].CreateDict()
                    options.PutNumber("FontSize", 15)
                    options.PutBool("UseSourceCodeFormatting", True)
                    options.PutNumber("PageWidth", 12)
                    options.PutNumber("PageHeight", 6)
                    Console.WriteLine("Converting from txt")
                    pdftron.PDF.Convert.FromText(pdfdoc, inputPath & "simple-text.txt", options)
                    pdfdoc.Save(outputPath & "simple-text.pdf", SDFDoc.SaveOptions.e_remove_unused)
                    Console.WriteLine("Saved simple-text.pdf")
                End Using

            Catch e As PDFNetException
                Console.WriteLine(e.Message)
                err = True
            End Try

            Try

                Using pdfdoc As PDFDoc = New PDFDoc(inputPath & "newsletter.pdf")
                    Console.WriteLine("Converting pdfdoc to SVG")
                    pdftron.PDF.Convert.ToSvg(pdfdoc, outputPath & "pdf2svg v2.svg")
                    Console.WriteLine("Saved pdf2svg v2.svg")
                End Using

            Catch e As PDFNetException
                Console.WriteLine(e.Message)
                err = True
            End Try

            Try
                Console.WriteLine("Converting PNG to XPS")
                pdftron.PDF.Convert.ToXps(inputPath & "butterfly.png", outputPath & "butterfly.xps")
                Console.WriteLine("Saved butterfly.xps")
            Catch e As PDFNetException
                Console.WriteLine(e.Message)
                err = True
            End Try

            Try
                Console.WriteLine("Converting PDF to XPS")
                pdftron.PDF.Convert.ToXps(inputPath & "newsletter.pdf", outputPath & "newsletter.xps")
                Console.WriteLine("Saved newsletter.xps")
            Catch e As PDFNetException
                Console.WriteLine(e.Message)
                err = True
            End Try

            Try
                Console.WriteLine("Converting PDF to HTML")
                pdftron.PDF.Convert.ToHtml(inputPath & "newsletter.pdf", outputPath & "newsletter")
                Console.WriteLine("Saved newsletter as HTML")
            Catch e As PDFNetException
                Console.WriteLine(e.Message)
                err = True
            End Try

            Try
                Console.WriteLine("Converting PDF to EPUB")
                pdftron.PDF.Convert.ToEpub(inputPath & "newsletter.pdf", outputPath & "newsletter.epub")
                Console.WriteLine("Saved newsletter.epub")
            Catch e As PDFNetException
                Console.WriteLine(e.Message)
                err = True
            End Try

            Try
                Console.WriteLine("Converting PDF to multipage TIFF")
                Dim tiff_options As pdftron.PDF.Convert.TiffOutputOptions = New pdftron.PDF.Convert.TiffOutputOptions()
                tiff_options.SetDPI(200)
                tiff_options.SetDither(True)
                tiff_options.SetMono(True)
                pdftron.PDF.Convert.ToTiff(inputPath & "newsletter.pdf", outputPath & "newsletter.tiff", tiff_options)
                Console.WriteLine("Saved newsletter.tiff")
            Catch e As PDFNetException
                Console.WriteLine(e.Message)
                err = True
            End Try

            Try

                Using pdfdoc As PDFDoc = New PDFDoc()
                    Console.WriteLine("Converting SVG to PDF")

                    pdftron.PDF.Convert.FromSvg(pdfdoc, inputPath & "tiger.svg", Nothing)
                    pdfdoc.Save(outputPath & "svg2pdf.pdf", SDFDoc.SaveOptions.e_remove_unused)

                    Console.WriteLine("Saved svg2pdf.pdf")
                End Using

            Catch e As PDFNetException
                Console.WriteLine(e.Message)
                err = True
            End Try

            Return err
        End Function

        Private Shared Function ConvertToPdfFromFile() As Boolean
            Dim testfiles As System.Collections.ArrayList = New System.Collections.ArrayList()
            testfiles.Add(New ConvertTestVB.Testfile("simple-powerpoint_2007.pptx", "pptx2pdf.pdf"))
            testfiles.Add(New ConvertTestVB.Testfile("simple-text.txt", "txt2pdf.pdf"))
            testfiles.Add(New ConvertTestVB.Testfile("simple-word_2007.docx", "docx2pdf.pdf"))
            testfiles.Add(New ConvertTestVB.Testfile("simple-excel_2007.xlsx", "xlsx2pdf.pdf"))
            testfiles.Add(New ConvertTestVB.Testfile("butterfly.png", "png2pdf.pdf"))
            testfiles.Add(New ConvertTestVB.Testfile("simple-xps.xps", "xps2pdf.pdf"))

            Dim err As Boolean = False

            For Each file As Testfile In testfiles
                Try

                    Using pdfdoc As pdftron.PDF.PDFDoc = New PDFDoc()

                        pdftron.PDF.Convert.Printer.SetMode(PDF.Convert.Printer.Mode.e_prefer_builtin_converter)
                        pdftron.PDF.Convert.ToPdf(pdfdoc, inputPath & file.inputFile)
                        pdfdoc.Save(outputPath & file.outputFile, SDFDoc.SaveOptions.e_linearized)
                        Console.WriteLine("Converted file: " & file.inputFile)
                        Console.WriteLine("to: " & file.outputFile)
                    End Using

                Catch e As PDFNetException
                    Console.WriteLine("ERROR: on input file " & file.inputFile)
                    Console.WriteLine(e.Message)
                    err = True
                End Try
            Next

            Return err
        End Function

        <STAThread>
        Shared Sub Main(ByVal args As String())
            PDFNet.Initialize(PDFTronLicense.Key)
            Dim err As Boolean = False
            err = ConvertToPdfFromFile()

            If err Then
                Console.WriteLine("ConvertFile failed")
            Else
                Console.WriteLine("ConvertFile succeeded")
            End If

            err = ConvertSpecificFormats()

            If err Then
                Console.WriteLine("ConvertSpecificFormats failed")
            Else
                Console.WriteLine("ConvertSpecificFormats succeeded")
            End If

            PDFNet.Terminate()
            Console.WriteLine("Done.")
        End Sub
    End Class
End Module
