'---------------------------------------------------------------------------------------
' Copyright (c) 2001-2025 by Apryse Software Inc. All Rights Reserved.
' Consult legal.txt regarding legal and license information.
'---------------------------------------------------------------------------------------
Imports System

Imports pdftron
Imports pdftron.Common
Imports pdftron.SDF
Imports pdftron.FDF
Imports pdftron.PDF

Module FDFTestVB
	Dim pdfNetLoader As PDFNetLoader
	Sub New()
		pdfNetLoader = pdftron.PDFNetLoader.Instance()
	End Sub

	'---------------------------------------------------------------------------------------
	' PDFNet includes full support for FDF (Forms Data Format) and for merging/extracting
	' forms data (FDF) with/from PDF. This sample illustrates basic FDF merge/extract functionality 
	' available in PDFNet.
	'---------------------------------------------------------------------------------------
	Sub Main()

		PDFNet.Initialize(PDFTronLicense.Key)

		' Relative path to the folder containing test files.
		Dim input_path As String = "../../../../TestFiles/"
		Dim output_path As String = "../../../../TestFiles/Output/"

		' Example 1)
		' Iterate over all form fields in the document. Display all field names.
		Try
			Using doc As PDFDoc = New PDFDoc(input_path + "form1.pdf")
				doc.InitSecurityHandler()

				Dim itr As FieldIterator = doc.GetFieldIterator()
				While itr.HasNext()
					Console.WriteLine("Field name: {0:s}", itr.Current().GetName())
					Console.WriteLine("Field partial name: {0:s}", itr.Current().GetPartialName())

					Console.Write("Field type: ")
					Dim type As Field.Type = itr.Current().GetType()
					If type = Field.Type.e_button Then
						Console.WriteLine("Button")
					ElseIf type = Field.Type.e_check Then
						Console.WriteLine("Check")
					ElseIf type = Field.Type.e_radio Then
						Console.WriteLine("Radio")
					ElseIf type = Field.Type.e_text Then
						Console.WriteLine("Text")
					ElseIf type = Field.Type.e_choice Then
						Console.WriteLine("Choice")
					ElseIf type = Field.Type.e_signature Then
						Console.WriteLine("Signature")
					ElseIf type = Field.Type.e_null Then
						Console.WriteLine("Null")
					End If
					Console.WriteLine("------------------------------")
					itr.Next()
				End While
			End Using
			Console.WriteLine("Done.")
		Catch e As Exception
			Console.WriteLine("Exception caught:\n{0}", e)
		End Try

		' Example 2) Import XFDF into FDF, then merge data from FDF into PDF
		Try
			' XFDF to FDF
			' form fields
			Console.WriteLine("Import form field data from XFDF to FDF.")
			
			Dim fdf_doc1 As FDFDoc = new FDFDoc(FDFDoc.CreateFromXFDF(input_path + "form1_data.xfdf"))
			fdf_doc1.Save(output_path + "form1_data.fdf")
			
			' annotations
			Console.WriteLine("Import annotations from XFDF to FDF.")
			
			Dim fdf_doc2 As FDFDoc = new FDFDoc(FDFDoc.CreateFromXFDF(input_path + "form1_annots.xfdf"))
			fdf_doc2.Save(output_path + "form1_annots.fdf")
			
			' FDF to PDF
			' form fields
			Console.WriteLine("Merge form field data from FDF.")
			
			Using doc As PDFDoc = New PDFDoc(input_path + "form1.pdf")
				doc.InitSecurityHandler()
				doc.FDFMerge(fdf_doc1)

				' Refreshing missing appearances is not required here, but is recommended to make them  
				' visible in PDF viewers with incomplete annotation viewing support. (such as Chrome)
				doc.RefreshAnnotAppearances()

				doc.Save(output_path + "form1_filled.pdf", SDF.SDFDoc.SaveOptions.e_linearized)

				' annotations
				Console.WriteLine("Merge annotations from FDF.")

				doc.FDFMerge(fdf_doc2)
				' Refreshing missing appearances is not required here, but is recommended to make them  
				' visible in PDF viewers with incomplete annotation viewing support. (such as Chrome)
				doc.RefreshAnnotAppearances()
				doc.Save(output_path + "form1_filled_with_annots.pdf", SDF.SDFDoc.SaveOptions.e_linearized)
			End Using

			Console.WriteLine("Done.")
		Catch e As Exception
			Console.WriteLine("Exception caught:\n{0}", e)
		End Try

		' Example 3) Extract data from PDF to FDF, then export FDF as XFDF
		Try
			' PDF to FDF
			Using in_doc As PDFDoc = New PDFDoc(output_path + "form1_filled_with_annots.pdf")
				in_doc.InitSecurityHandler()

				' form fields only
				Console.WriteLine("Extract form fields data to FDF.")

				Dim doc_fields As FDFDoc = in_doc.FDFExtract(PDF.PDFDoc.ExtractFlag.e_forms_only)
				doc_fields.SetPdfFileName("../form1_filled_with_annots.pdf")
				doc_fields.Save(output_path + "form1_filled_data.fdf")

				' annotations only
				Console.WriteLine("Extract annotations to FDF.")

				Dim doc_annots As FDFDoc = in_doc.FDFExtract(PDF.PDFDoc.ExtractFlag.e_annots_only)
				doc_annots.SetPdfFileName("../form1_filled_with_annots.pdf")
				doc_annots.Save(output_path + "form1_filled_annot.fdf")

				' both form fields and annotations
				Console.WriteLine("Extract both form fields and annotations to FDF.")

				Dim doc_both As FDFDoc = in_doc.FDFExtract(PDF.PDFDoc.ExtractFlag.e_both)
				doc_both.SetPdfFileName("../form1_filled_with_annots.pdf")
				doc_both.Save(output_path + "form1_filled_both.fdf")

				' FDF to XFDF
				' form fields
				Console.WriteLine("Export form field data from FDF to XFDF.")

				doc_fields.SaveAsXFDF(output_path + "form1_filled_data.xfdf")

				' annotations
				Console.WriteLine("Export annotations from FDF to XFDF.")

				doc_annots.SaveAsXFDF(output_path + "form1_filled_annot.xfdf")

				' both form fields and annotations
				Console.WriteLine("Export both form fields and annotations from FDF to XFDF.")

				doc_both.SaveAsXFDF(output_path + "form1_filled_both.xfdf")
			End Using
			Console.WriteLine("Done.")
		Catch e As Exception
			Console.WriteLine("Exception caught:\n{0}", e)
		End Try

		' Example 4) Merge/Extract XFDF into/from PDF
		Try
			' Merge XFDF from string
			Using in_doc As PDFDoc = New PDFDoc(input_path + "numbered.pdf")
				in_doc.InitSecurityHandler()

				Console.WriteLine("Merge XFDF string into PDF.")
				Dim str As String = "<?xml version=""1.0"" encoding=""UTF-8"" ?><xfdf xmlns=""http://ns.adobe.com/xfdf"" xml:space=""preserve""><square subject=""Rectangle"" page=""0"" name=""cf4d2e58-e9c5-2a58-5b4d-9b4b1a330e45"" title=""user"" creationdate=""D:20120827112326-07'00'"" date=""D:20120827112326-07'00'"" rect=""227.7814207650273,597.6174863387978,437.07103825136608,705.0491803278688"" color=""#000000"" interior-color=""#FFFF00"" flags=""print"" width=""1""><popup flags=""print,nozoom,norotate"" open=""no"" page=""0"" rect=""0,792,0,792"" /></square></xfdf>"

				Dim fdoc As FDFDoc = New FDFDoc(FDFDoc.CreateFromXFDF(str))
				in_doc.FDFMerge(fdoc)
				in_doc.Save(output_path + "numbered_modified.pdf", SDF.SDFDoc.SaveOptions.e_linearized)
				Console.WriteLine("Merge complete.")

				' Extract XFDF as string
				Console.WriteLine("Extract XFDF as a string.")
				Dim fdoc_new As FDFDoc = in_doc.FDFExtract(PDF.PDFDoc.ExtractFlag.e_both)
				Dim XFDF_str As String = fdoc_new.SaveAsXFDF()
				Console.WriteLine("Extracted XFDF: ")
				Console.WriteLine(XFDF_str)
			End Using
			Console.WriteLine("Extract complete.")

		Catch e As Exception
			Console.WriteLine("Exception caught:\n{0}", e)
		End Try

		' Example 5) Read FDF files directly
		Try
			Dim doc As FDFDoc = New FDFDoc(output_path + "form1_filled_data.fdf")

			Dim itr As FDFFieldIterator = doc.GetFieldIterator()
			While itr.HasNext()
				Console.WriteLine("Field name: {0:s}", itr.Current().GetName())
				Console.WriteLine("Field partial name: {0:s}", itr.Current().GetPartialName())
				Console.WriteLine("------------------------------")
				itr.Next()
			End While

			Console.WriteLine("Done.")
		Catch e As Exception
			Console.WriteLine("Exception caught:\n{0}", e)
		End Try

		' Example 6) Direct generation of FDF.
		Try
			Dim doc As FDFDoc = New FDFDoc

			' Create new fields (i.e. key/value pairs).
			doc.FieldCreate("Company", Int(Field.Type.e_text), "PDFTron Systems")
			doc.FieldCreate("First Name", Int(Field.Type.e_text), "John")
			doc.FieldCreate("Last Name", Int(Field.Type.e_text), "Doe")
			' ...		

			' doc.SetPdfFileName("mydoc.pdf");
			doc.Save(output_path + "sample_output.fdf")
			Console.WriteLine("Done. Results saved in sample_output.fdf")
		Catch e As Exception
			Console.WriteLine("Exception caught:\n{0}", e)
		End Try
		PDFNet.Terminate()
	End Sub
End Module
