'
' Copyright (c) 2001-2025 by Apryse Software Inc. All Rights Reserved.
'

Imports System

Imports pdftron
Imports pdftron.Common
Imports pdftron.Filters
Imports pdftron.SDF
Imports pdftron.PDF
'---------------------------------------------------------------------------------------
' The following sample illustrates how to convert HTML pages to PDF format using
' the HTML2PDF class.
' 
' 'pdftron.PDF.HTML2PDF' is an optional PDFNet Add-On utility class that can be 
' used to convert HTML web pages into PDF documents by using an external module (html2pdf).
'
' html2pdf modules can be downloaded from http://www.pdftron.com/pdfnet/downloads.html.
'
' Users can convert HTML pages to PDF using the following operations:
' - Simple one line static method to convert a single web page to PDF. 
' - Convert HTML pages from URL or string, plus optional table of contents, in user defined order. 
' - Optionally configure settings for proxy, images, java script, and more for each HTML page. 
' - Optionally configure the PDF output, including page size, margins, orientation, and more. 
' - Optionally add table of contents, including setting the depth and appearance.
'---------------------------------------------------------------------------------------
Module HTML2PDFTestVB
    Dim pdfNetLoader As PDFNetLoader
    Sub New()
        pdfNetLoader = pdftron.PDFNetLoader.Instance()
    End Sub

    Sub Main()
        Dim output_path As String = "../../../../TestFiles/Output/html2pdf_example"
        Dim host As String = "https://docs.apryse.com"
        Dim page0 As String = "/"
        Dim page1 As String = "/all-products/"
        Dim page2 As String =  "/documentation/web/faq"
        ' The first step in every application using PDFNet is to initialize the 
        ' library and set the path to common PDF resources. The library is usually 
        ' initialized only once, but calling Initialize() multiple times is also fine.

        PDFNet.Initialize(PDFTronLicense.Key)

        ' For HTML2PDF we need to locate the html2pdf module. If placed with the 
        ' PDFNet library, or in the current working directory, it will be loaded
        ' automatically. Otherwise, it must be set manually using HTML2PDF.SetModulePath().
        HTML2PDF.SetModulePath("../../../../../Lib")

        If Not HTML2PDF.IsModuleAvailable() Then
            Console.WriteLine()
            Console.WriteLine("Unable to run HTML2PDFTest: Apryse SDK CAD module not available.")
            Console.WriteLine("---------------------------------------------------------------")
            Console.WriteLine("The HTML2PDF module is an optional add-on, available for download")
            Console.WriteLine("at http://www.pdftron.com/. If you have already downloaded this")
            Console.WriteLine("module, ensure that the SDK is able to find the required files")
            Console.WriteLine("using the HTML2PDF.SetModulePath() function.")
            Console.WriteLine()
        End If
        '--------------------------------------------------------------------------------
        ' Example 1) Simple conversion of a web page to a PDF doc. 

        Try
            Dim doc As PDFDoc = New PDFDoc()
            HTML2PDF.Convert(doc, host & page0)
            doc.Save(output_path + "_01.pdf", SDFDoc.SaveOptions.e_linearized)
        Catch ex As PDFNetException
            Console.WriteLine(ex.Message)
        Catch ex As Exception
            MsgBox(ex.Message)
        End Try

        '--------------------------------------------------------------------------------
        ' Example 2) Modify the settings of the generated PDF pages and attach to an
        ' existing PDF document. 

        Try
            ' open the existing PDF, and initialize the security handler
            Dim doc As PDFDoc = New PDFDoc("../../../../TestFiles/numbered.pdf")
            doc.InitSecurityHandler()

            ' create the HTML2PDF converter object and modify the output of the PDF pages
            Dim converter As HTML2PDF = New HTML2PDF()
            converter.SetPaperSize(PrinterMode.PaperSize.e_11x17)

            ' insert the web page to convert
            converter.InsertFromURL(host & page0)

            ' convert the web page, appending generated PDF pages to doc
            converter.Convert(doc)
            doc.Save(output_path + "_02.pdf", SDFDoc.SaveOptions.e_linearized)
        Catch ex As PDFNetException
            Console.WriteLine(ex.Message)
        Catch ex As Exception
            MsgBox(ex.Message)
        End Try

        '--------------------------------------------------------------------------------
        ' Example 3) Convert multiple web pages

        Try
            Dim doc As PDFDoc = New PDFDoc()
            ' convert page 0 into pdf
            Dim converter As HTML2PDF = New HTML2PDF()
            Dim header As String = "<div style='width:15%;margin-left:0.5cm;text-align:left;font-size:10px;color:#0000FF'><span class='date'></span></div><div style='width:70%;direction:rtl;white-space:nowrap;overflow:hidden;text-overflow:clip;text-align:center;font-size:10px;color:#0000FF'><span>PDFTRON HEADER EXAMPLE</span></div><div style='width:15%;margin-right:0.5cm;text-align:right;font-size:10px;color:#0000FF'><span class='pageNumber'></span> of <span class='totalPages'></span></div>"
            Dim footer As String = "<div style='width:15%;margin-left:0.5cm;text-align:left;font-size:7px;color:#FF00FF'><span class='date'></span></div><div style='width:70%;direction:rtl;white-space:nowrap;overflow:hidden;text-overflow:clip;text-align:center;font-size:7px;color:#FF00FF'><span>PDFTRON FOOTER EXAMPLE</span></div><div style='width:15%;margin-right:0.5cm;text-align:right;font-size:7px;color:#FF00FF'><span class='pageNumber'></span> of <span class='totalPages'></span></div>"
            converter.SetHeader(header)
            converter.SetFooter(footer)
            converter.SetMargins("1cm", "2cm", ".5cm", "1.5cm")
            Dim settings As HTML2PDF.WebPageSettings = New HTML2PDF.WebPageSettings()
            settings.SetZoom(0.5)
            converter.InsertFromURL(host & page0, settings)
            converter.Convert(doc)

            ' convert page 1 with the same settings, appending generated PDF pages to doc
            converter.InsertFromURL(host & page1, settings)
            converter.Convert(doc)

            ' convert page 2 with different settings, appending generated PDF pages to doc
            Dim another_converter As HTML2PDF = New HTML2PDF()
            another_converter.SetLandscape(True)
            Dim another_settings As HTML2PDF.WebPageSettings = New HTML2PDF.WebPageSettings()
            another_settings.SetPrintBackground(False)
            another_converter.InsertFromURL(host & page2, another_settings)
            another_converter.Convert(doc)

            doc.Save(output_path + "_03.pdf", SDFDoc.SaveOptions.e_linearized)
        Catch ex As PDFNetException
            Console.WriteLine(ex.Message)
        Catch ex As Exception
            MsgBox(ex.Message)
        End Try

        '--------------------------------------------------------------------------------
        ' Example 4) Convert HTML string to PDF. 

        Try
            Dim doc As PDFDoc = New PDFDoc()

            Dim converter As HTML2PDF = New HTML2PDF()

            ' Our HTML data
            Dim html As String = "<html><body><h1>Heading</h1><p>Paragraph.</p></body></html>"

            ' Add html data
            converter.InsertFromHtmlString(html)
            ' Note, InsertFromHtmlString can be mixed with the other Insert methods.

            converter.Convert(doc)
            doc.Save(output_path + "_04.pdf", SDFDoc.SaveOptions.e_linearized)
        Catch ex As PDFNetException
            Console.WriteLine(ex.Message)
        Catch ex As Exception
            MsgBox(ex.Message)
        End Try

        '--------------------------------------------------------------------------------
        ' Example 5) Set the location of the log file to be used during conversion. 

        Try
            Dim doc As PDFDoc = New PDFDoc()
            Dim converter As HTML2PDF = New HTML2PDF()
            converter.SetLogFilePath("../../../../TestFiles/Output/html2pdf.log")
            converter.InsertFromURL(host & page0)
            converter.Convert(doc)
            doc.Save(output_path + "_05.pdf", SDFDoc.SaveOptions.e_linearized)
        Catch ex As PDFNetException
            Console.WriteLine(ex.Message)
        Catch ex As Exception
            MsgBox(ex.Message)
        End Try

        PDFNet.Terminate()
    End Sub

End Module
