'
' Copyright (c) 2001-2025 by Apryse Software Inc. All Rights Reserved.
'

Imports System
Imports System.Collections.Generic
Imports pdftron
Imports pdftron.Common
Imports pdftron.Filters
Imports pdftron.SDF
Imports PDFTRON.PDF

' This sample illustrates the basic text highlight capabilities of PDFNet.
' It simulates a full-text search engine that finds all occurrences of the word 'Federal'.
' It then highlights those words on the page.
'
' Note: The TextSearch class is the preferred solution for searching text within a single
' PDF file. TextExtractor provides search highlighting capabilities where a large number
' of documents are indexed using a 3rd party search engine.

Module HighlightsTestVB
	Dim pdfNetLoader As PDFNetLoader
	Sub New()
		pdfNetLoader = pdftron.PDFNetLoader.Instance()
	End Sub

	Sub Main()

		PDFNet.Initialize(PDFTronLicense.Key)

		' Relative path to the folder containing test files.
		Dim input_path As String = "../../../../TestFiles/"
		Dim output_path As String = "../../../../TestFiles/Output/"

		' Sample code showing how to use high-level text highlight APIs.
		Try
			Using doc As PDFDoc = New PDFDoc(input_path + "paragraphs_and_tables.pdf")
				doc.InitSecurityHandler()

				Dim pg As Page = doc.GetPage(1)
				If pg Is Nothing Then
					Console.WriteLine("Page not found.")
					PDFNet.Terminate()
					Return
				End If

				Using txt As New TextExtractor
					txt.Begin(pg)  ' read the page

					' Do not dehyphenate; that would interfere with character offsets
					Dim dehyphen As Boolean = False
					' Retrieve the page text
					Dim page_text As String = txt.GetAsText(dehyphen)

					' Simulating a full-text search engine that finds all occurrences of the word 'Federal'.
					' In a real application, plug in your own search engine here.
					Dim search_text As String = "Federal"
					Dim char_ranges_list As New List(Of TextExtractor.CharRange)()
					Dim ofs As Integer = page_text.IndexOf(search_text)
					While ofs >= 0
						char_ranges_list.Add(New TextExtractor.CharRange(ofs, search_text.Length))  ' character offset + length
						ofs = page_text.IndexOf(search_text, ofs + 1)
					End While

					' Convert List to array, as required by TextExtractor.GetHighlights()
					Dim char_ranges As TextExtractor.CharRange() = char_ranges_list.ToArray()

					' Retrieve Highlights object and apply annotations to the page
					Dim hlts As Highlights = txt.GetHighlights(char_ranges)
					hlts.Begin(doc)
					While hlts.HasNext()
						Dim quads As Double() = hlts.GetCurrentQuads()
						Dim quad_count As Integer = quads.Length / 8

						For i As Integer = 0 To quad_count - 1
							' assume each quad is an axis-aligned rectangle
							Dim offset As Integer = 8 * i
							Dim x1 As Double = Math.Min(Math.Min(Math.Min(quads(offset + 0), quads(offset + 2)), quads(offset + 4)), quads(offset + 6))
							Dim x2 As Double = Math.Max(Math.Max(Math.Max(quads(offset + 0), quads(offset + 2)), quads(offset + 4)), quads(offset + 6))
							Dim y1 As Double = Math.Min(Math.Min(Math.Min(quads(offset + 1), quads(offset + 3)), quads(offset + 5)), quads(offset + 7))
							Dim y2 As Double = Math.Max(Math.Max(Math.Max(quads(offset + 1), quads(offset + 3)), quads(offset + 5)), quads(offset + 7))
							Dim highlight As pdftron.PDF.Annots.Highlight = pdftron.PDF.Annots.Highlight.Create(doc, New Rect(x1, y1, x2, y2))
							highlight.RefreshAppearance()
							pg.AnnotPushBack(highlight)

							Console.WriteLine("[{0:N2}, {1:N2}, {2:N2}, {3:N2}]", x1, y1, x2, y2)
						Next

						hlts.Next()
					End While

					' Output highlighted PDF doc
					doc.Save(output_path & "search_highlights.pdf", SDFDoc.SaveOptions.e_linearized)
				End Using
			End Using
		Catch ex As PDFNetException
			Console.WriteLine(ex.Message)
		Catch ex As Exception
			MsgBox(ex.Message)
		End Try

		PDFNet.Terminate()
	End Sub

End Module
