//
// Copyright (c) 2001-2025 by Apryse Software Inc. All Rights Reserved.
//

using System;
using System.Drawing;
using System.Drawing.Drawing2D;

using pdftron;
using pdftron.Common;
using pdftron.Filters;
using pdftron.SDF;
using pdftron.PDF;

namespace PDFDCTestCS
{
	/// <summary>
	/// This sample shows how to create and use PDFDC (i.e. a PDF Device Context).
	/// Windows developers can use standard GDI or GDIPlus API-s to write on PDFDC 
	/// and to generate PDF documents based on their existing drawing functions.
	/// 
	/// The second portion of this sample shows how to create and use PDFDCEX.
	/// Windows developers can use standard GDI or GDIPlus API-s to write multi-page
	/// PDF documents using existing drawing code.
	///
	/// PDFDCEX can also be used to implement file conversion from any printable 
	/// file format to PDF (i.e. a virtual PDF printer driver).
	/// </summary>
	class Class1
	{
		private static pdftron.PDFNetLoader pdfNetLoader = pdftron.PDFNetLoader.Instance();
		static Class1() {}
		
		// Relative path to the folder containing test files.
		const string input_path = "../../../../TestFiles/";
		const string output_path = "../../../../TestFiles/Output/";

		/// <summary>
		/// The main entry point for the application.
		/// </summary>
		[STAThread]
		static void Main(string[] args)
		{
			PDFNet.Initialize(PDFTronLicense.Key);

			try
			{
				//////////////////////////////////////////////////////////////////////////
				// First the PDFDC
				// Start with a PDFDoc to put the picture into, and a PDFDC to translate GDI to PDF
				using (PDFDoc pdfdoc = new PDFDoc())
				{
					PDFDC pdf_dc = new PDFDC();

					// Set the scale between GDI logical units and the PDF page at 50/inch.
					pdf_dc.SetDPI(50);

					// Create a page to put the GDI content onto
					Page page = pdfdoc.PageCreate();

					// Begin the translation from GDI to PDF.
					// Provide the page to place the picture onto, and the bounding box for the content.
					// We're going to scale the GDI content to fill the page while preserving the aspect
					// ratio.
					// Get back a GDIPlus Graphics Object
					using (Graphics gr = pdf_dc.Begin(page, page.GetCropBox()))
					{
						// Create a path that consists of a single polygon.
						System.Drawing.Point[] polyPoints = 
						{
							new System.Drawing.Point(10, 10),
							new System.Drawing.Point(150, 10), 
							new System.Drawing.Point(100, 75),
							new System.Drawing.Point(100, 150)
						};

						GraphicsPath path = new GraphicsPath();
						path.AddPolygon(polyPoints);

						// Construct a region based on the path.
						Region region = new Region(path);

						// Draw the outline of the region.
						Pen pen = Pens.Black;
						gr.DrawPath(pen, path);

						// Set the clipping region of the Graphics object.
						gr.SetClip(region, CombineMode.Replace);

						// Draw some clipped strings.
						FontFamily fontFamily = new FontFamily("Arial");
						System.Drawing.Font font = new System.Drawing.Font(
							fontFamily,
							36, FontStyle.Bold,
							GraphicsUnit.Pixel);
						SolidBrush solidBrush = new SolidBrush(Color.FromArgb(255, 255, 0, 0));

						gr.DrawString(
							"A Clipping Region",
							font, solidBrush,
							new PointF(15, 25));

						gr.DrawString(
							"A Clipping Region",
							font,
							solidBrush,
							new PointF(15, 68));


						pdf_dc.End(); // Close PDF Device Context
					}

					// Add the page to the document
					pdfdoc.PagePushBack(page);

					//////////////////////////////////////////////////////////////////////////
					// Page two
					page = pdfdoc.PageCreate();
					using (Graphics gr = pdf_dc.Begin(page, page.GetCropBox(), true))
					{
						Rectangle myRectangle = new Rectangle(0, 0, 100, 50);
						LinearGradientBrush myLinearGradientBrush = new LinearGradientBrush(
							myRectangle,
							Color.Blue,
							Color.Green,
							LinearGradientMode.Horizontal);
						gr.FillEllipse(myLinearGradientBrush, myRectangle);

						// Complete the translation
						pdf_dc.End();
					}

					// Add the page to the document
					pdfdoc.PagePushBack(page);

					pdfdoc.Save(output_path + "PDFDCTest.pdf", SDFDoc.SaveOptions.e_remove_unused);
				}
				Console.WriteLine("Saved PDFDCTest.pdf");


				//////////////////////////////////////////////////////////////////////////
				// 
				// Now for PDFDCEX
				// Start with a PDFDoc to put the picture into, and a PDFDCEX to translate GDI to PDF
				using (PDFDoc pdfdoc2 = new PDFDoc())
				{
					PDFDCEX pdf_dcex = new PDFDCEX();

					// Begin the translation from GDI to PDF.
					// The page dimensions and the converter DPI determine the coordinate system.
					// Unlike PDFDC, the drawings will not be scaled to fit the bounding box.
					pdf_dcex.Begin(pdfdoc2);

					// Start the first page -- Get back a Graphics Object
					System.Drawing.Graphics gr2 = pdf_dcex.StartPage();

					// Create a path that consists of a single polygon.
					System.Drawing.Point[] polyPoints2 = 
					{
						new System.Drawing.Point(10, 10),
						new System.Drawing.Point(150, 10), 
						new System.Drawing.Point(100, 75),
						new System.Drawing.Point(100, 150)
					};

					GraphicsPath path2 = new GraphicsPath();
					path2.AddPolygon(polyPoints2);

					// Construct a region based on the path.
					Region region2 = new Region(path2);

					// Draw the outline of the region.
					Pen pen2 = Pens.Black;
					gr2.DrawPath(pen2, path2);

					// Set the clipping region of the Graphics object.
					gr2.SetClip(region2, CombineMode.Replace);

					// Draw some clipped strings.
					FontFamily fontFamily2 = new FontFamily("Arial");
					System.Drawing.Font font2 = new System.Drawing.Font(
						fontFamily2,
						36, FontStyle.Bold,
						GraphicsUnit.Pixel);
					SolidBrush solidBrush2 = new SolidBrush(Color.FromArgb(255, 255, 0, 0));

					gr2.DrawString(
						"A Clipping Region",
						font2, solidBrush2,
						new PointF(15, 25));

					gr2.DrawString(
						"A Clipping Region",
						font2,
						solidBrush2,
						new PointF(15, 68));

					pdf_dcex.EndPage();
					// at this point the Graphics Context gr2 has been disposed

					//////////////////////////////////////////////////////////////////////////
					// Page two -- get a new Graphics Object

					gr2 = pdf_dcex.StartPage();

					Rectangle myRectangle2 = new Rectangle(0, 0, 100, 50);
					LinearGradientBrush myLinearGradientBrush2 = new LinearGradientBrush(
						myRectangle2,
						Color.Blue,
						Color.Green,
						LinearGradientMode.Horizontal);
					gr2.FillEllipse(myLinearGradientBrush2, myRectangle2);
					gr2.DrawString(
						"An Ellipse Filled with a Linear Gradient",
						font2, solidBrush2,
						new PointF(15, 200));
					pdf_dcex.EndPage();

					// Complete the translation
					pdf_dcex.End();

					pdfdoc2.Save(output_path + "PDFDCEXTest.pdf", SDFDoc.SaveOptions.e_remove_unused);
				}
				Console.WriteLine("Saved PDFDCEXTest.pdf");
				Console.WriteLine("Done.");
			}
			catch (PDFNetException e)
			{
				Console.WriteLine(e.Message);
			}
			PDFNet.Terminate();
		}
	}
}
