'
' Copyright (c) 2001-2025 by Apryse Software Inc. All Rights Reserved.
'
Imports System
Imports System.Drawing
Imports System.Drawing.Printing

Imports pdftron
Imports pdftron.Common
Imports pdftron.Filters
Imports pdftron.SDF
Imports pdftron.PDF

' The following sample illustrates how to print PDF document using currently selected
' default printer. 
' 
' The first example uses the new PDF::Print::StartPrintJob function to send a rasterization 
' of the document with optimal compression to the printer.  If the OS is Windows 7, then the
' XPS print path will be used to preserve vector quality.' 
'
' The second example uses PDFDraw send unoptimized rasterized data. 
' 
' If you would like to rasterize page at high resolutions (e.g. more than 600 DPI), you 
' should use PDFRasterizer or PDFNet vector output instead of PDFDraw. 

Namespace PDFViewSimpleTestVB
    Module PDFViewSimpleTestVB
        Dim pdfNetLoader As PDFNetLoader
        Sub New()
            pdfNetLoader = pdftron.PDFNetLoader.Instance()
        End Sub

        Sub Main()
            Dim driver As PDFPrint = New PDFPrint
            driver.Execute()
        End Sub
    End Module
End Namespace

Public Class PDFPrint
    Private doc As PDFDoc = Nothing
    Private pdfdraw As PDFDraw = Nothing
    Private pageitr As PageIterator = Nothing

    Sub Execute()
        PDFNet.Initialize(PDFTronLicense.Key)

        ' Optional: Set ICC color profiles to fine tune color conversion 
        ' for PDF 'device' color spaces. You can use your own ICC profiles. 
        ' Standard Adobe color profiles can be download from Adobes site: 
        ' http://www.adobe.com/support/downloads/iccprofiles/iccprofiles_win.html
        '
        ' Simply drop all *.icc files in PDFNet resource folder or you specify 
        ' the full pathname.
        '---
        ' Try
        ' PDFNet.SetColorManagement()
        ' PDFNet.SetDefaultDeviceCMYKProfile("USWebCoatedSWOP.icc") ' will search in PDFNet resource folder.
        ' PDFNet.SetDefaultDeviceRGBProfile("AdobeRGB1998.icc") 

        ' Optional: Set predefined font mappings to override default font 
        ' substitution for documents with missing fonts. For example:
        '---
        ' PDFNet.AddFontSubst("StoneSans-Semibold", "C:/WINDOWS/Fonts/comic.ttf")
        ' PDFNet.AddFontSubst("StoneSans", "comic.ttf")  ' search for 'comic.ttf' in PDFNet resource folder.
        ' PDFNet.AddFontSubst(PDFNet.CharacterOrdering.e_Identity, "C:/WINDOWS/Fonts/arialuni.ttf")
        ' PDFNet.AddFontSubst(PDFNet.CharacterOrdering.e_Japan1, "C:/Program Files/Adobe/Acrobat 7.0/Resource/CIDFont/KozMinProVI-Regular.otf")
        ' PDFNet.AddFontSubst(PDFNet.CharacterOrdering.e_Japan2, "c:/myfonts/KozMinProVI-Regular.otf")
        '
        ' If fonts are in PDFNet resource folder, it is not necessary to specify 
        ' the full path name. For example,
        '---
        ' PDFNet.AddFontSubst(PDFNet.CharacterOrdering.e_Korea1, "AdobeMyungjoStd-Medium.otf")
        ' PDFNet.AddFontSubst(PDFNet.CharacterOrdering.e_CNS1, "AdobeSongStd-Light.otf")
        ' PDFNet.AddFontSubst(PDFNet.CharacterOrdering.e_GB1, "AdobeMingStd-Light.otf")
        ' Catch e As PDFNetException
        '   Console.WriteLine(e.Message)
        ' End Try


        ' Relative path to the folder containing test files.
        Dim input_path As String = "../../../../TestFiles/"
        Try
            ' Open the PDF document.
            Console.WriteLine("Opening the input file...")
            doc = New PDFDoc(input_path + "tiger.pdf")
            doc.InitSecurityHandler()

            ''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
            ' Example 1: use the PDF::Print::StartPrintJob interface
            ' This is silent (no progress dialog) and blocks until print job is at spooler
            ' The rasterized print job is compressed before sending to printer
            Console.WriteLine("Printing the input file using PDF.Print.StartPrintJob...")

            ' While you can get the name of the default printer by using:
            ' Dim ps As PrinterSettings = New PrinterSettings
            ' Dim printerName As String = ps.PrinterName()
            ' Print.StartPrintJob can also determine this for you, just pass an empty printer name

            ' To setup printing options:
            Dim printerMode As PrinterMode = New PrinterMode
            printerMode.SetCollation(True)
            printerMode.SetCopyCount(1)
            printerMode.SetDPI(300)     ' regardless of ordering, an explicit DPI setting overrides the OutputQuality setting
            printerMode.SetDuplexing(printerMode.DuplexMode.e_Duplex_Auto)
            printerMode.SetOutputColor(printerMode.OutputColor.e_OutputColor_Grayscale)
            printerMode.SetOutputQuality(printerMode.OutputQuality.e_OutputQuality_Medium)
            Dim pagesToPrint As PageSet = New PageSet(1, doc.GetPageCount(), PageSet.Filter.e_all)

            ' Print the document on the default printer, name the print job the name of the 
            ' file, print to the printer not a file, and use default printer options:
            Print.StartPrintJob(doc, "", doc.GetFileName(), "", pagesToPrint, printerMode, Nothing)


            ''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
            ' Example 2: use the .Net PrintDocument class and PDFDraw rasterizer
            ' This will pop up a progress dialog

            ' Start printing from the first page
            pageitr = doc.GetPageIterator()

            pdfdraw = New PDFDraw
            pdfdraw.SetPrintMode(True)
            pdfdraw.SetRasterizerType(PDFRasterizer.Type.e_BuiltIn)

            ' Create a printer
            Dim printer As PrintDocument = New PrintDocument

            ' Set the PrintPage delegate which will be invoked to print each page
            AddHandler printer.PrintPage, AddressOf PrintPage
            printer.Print()          ' Start printing

            pdfdraw.Dispose()            ' Free allocated resources (generally a good idea when printing many documents).
            doc.Close()           ' Close the file
        Catch e As PDFNetException
            Console.WriteLine(e.Message)
        End Try
        PDFNet.Terminate()
    End Sub

    Private Sub PrintPage(ByVal sender As Object, ByVal ev As PrintPageEventArgs)
        Dim gr As Graphics = ev.Graphics
        gr.PageUnit = GraphicsUnit.Inch
        Dim rectPage As Rectangle = ev.PageBounds

        Dim dpi As Single = gr.DpiX
        If dpi > 300 Then dpi = 300
        pdfdraw.SetDPI(dpi)

        Dim hardMarginX As Integer = 0
        Dim hardMarginY As Integer = 0
        Dim left As Double = (rectPage.Left - hardMarginX) / 100
        Dim right As Double = (rectPage.Right - hardMarginX) / 100
        Dim top As Double = (rectPage.Top - hardMarginY) / 100
        Dim bottom As Double = (rectPage.Bottom - hardMarginY) / 100
        Dim rect As pdftron.PDF.Rect = New Rect(left * 72, bottom * 72, right * 72, top * 72)

        Try
            pdfdraw.DrawInRect(pageitr.Current, gr, rect)
        Catch ex As Exception
            Console.WriteLine("Printing Error: " + ex.ToString)
        End Try

        pageitr.Next()
        ev.HasMorePages = pageitr.HasNext()
    End Sub
End Class
