﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Navigation;
using System.Windows.Shapes;
using System.Globalization;

namespace PDFViewWPFTestCS.Controls
{
    /// <summary>
    /// Interaction logic for ColorPicker.xaml
    /// </summary>
    public partial class ColorPicker : UserControl
    {
        public ColorPicker()
        {
            InitializeComponent();
            colorlistbox.SelectionChanged += new SelectionChangedEventHandler(colorlistbox_SelectionChanged);
        }

        #region DependencyProperty

        public Color Value
        {
            get { return (Color)GetValue(ValueProperty); }
            set { SetValue(ValueProperty, value); }
        }

        public event RoutedEventHandler ValueChanged
        {
            add { AddHandler(ValueChangedEvent, value); }
            remove { RemoveHandler(ValueChangedEvent, value); }
        }

        public static readonly DependencyProperty ValueProperty = DependencyProperty.Register("Value", typeof(Color), typeof(ColorPicker));
        public static readonly RoutedEvent ValueChangedEvent = EventManager.RegisterRoutedEvent("ValueChanged", RoutingStrategy.Bubble, typeof(RoutedEventHandler), typeof(ColorPicker));

        #endregion

        void colorlistbox_SelectionChanged(object sender, SelectionChangedEventArgs e)
        {
            e.Handled = true;

            ColorConverter cc = new ColorConverter();
            Color c = new Color();
            c = (Color)cc.ConvertBack(colorlistbox.SelectedValue, c.GetType(), null, CultureInfo.CurrentCulture);
            Value = c;
            RoutedEventArgs args = new RoutedEventArgs(ValueChangedEvent);
            RaiseEvent(args);
        }

        
    }

    /// <summary>
    /// A class that converts a <c>Windows.Media.Color</c> object to a string representation and vice versa.
    /// </summary>
    [ValueConversion(typeof(Color), typeof(String))]
    public class ColorConverter : IValueConverter
    {
        /// <summary>
        /// Converts a <c>Windows.Media.Color?</c> object into a its string representation: ARGB hex values (i.e #FF000000).
        /// </summary>
        /// <param name="value">A nullable color object</param>
        /// <param name="targetType"></param>
        /// <param name="parameter"></param>
        /// <param name="culture"></param>
        /// <returns>A string representation of the color's ARGB hex values</returns>
        public object Convert(object value, Type targetType, object parameter, System.Globalization.CultureInfo culture)
        {
            var color = (Color)(value as Color?);

            //If Alpha is 00, return Transparent
            if (color.A.Equals(0)) return "Transparent";

            return "#" + color.R.ToString("X2") + color.G.ToString("X2") + color.B.ToString("X2");
        }

        /// <summary>
        /// A ARGB hex color string to a nullable Windows.Media.Color object
        /// </summary>
        public object ConvertBack(object value, Type targetType, object parameter, CultureInfo culture)
        {
            var colorString = value as string;

            if (String.IsNullOrWhiteSpace(colorString)) return Colors.Transparent;
            if (colorString.Equals("Transparent")) return Colors.Transparent;

            byte a = 255;
            byte r = (byte)(System.Convert.ToUInt32(colorString.Substring(1, 2), 16));
            byte g = (byte)(System.Convert.ToUInt32(colorString.Substring(3, 2), 16));
            byte b = (byte)(System.Convert.ToUInt32(colorString.Substring(5, 2), 16));

            return Color.FromArgb(a, r, g, b);
        }
    }
}
