﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Navigation;
using System.Windows.Shapes;
using System.Windows.Controls.Primitives;
using System.Runtime.InteropServices;
using System.Printing;

using pdftron;
using pdftron.PDF;
using pdftron.SDF;
using pdftron.Filters;
using pdftron.Common;
using System.ComponentModel;

namespace PDFViewWPFTestCS
{
    /// <summary>
    /// Interaction logic for MainWindow.xaml
    /// </summary>
    public partial class MainWindow : Window
    {
        private static pdftron.PDFNetLoader pdfNetLoader = pdftron.PDFNetLoader.Instance();
        static MainWindow() {}

        private bool _sliderShouldForwardValue = true;
        private PDFViewViewer _currentViewer;
        private TabItem _currentTab;

        FindTextDialog _findTextDialog;
        public FindTextDialog FindTextDialog
        {
            get { return _findTextDialog; }
            set { _findTextDialog = value; }
        }

        public struct PrinterArguments
        {
            public PDFDoc DocToPrint;
            public string PrinterName;
            public string FileName;
            public PageSet PageSet;
            public PrinterMode PrinterMode;
            public pdftron.PDF.OCG.Context OCG;

            public PrinterArguments(PDFDoc docToPrint, string printerName, string fileName, PageSet pageSet, PrinterMode printerMode, pdftron.PDF.OCG.Context ocg)
            {
                this.DocToPrint = docToPrint;
                this.PrinterName = printerName;
                this.FileName = fileName;
                this.PageSet = pageSet;
                this.PrinterMode = printerMode;
                this.OCG = ocg;
            }
        }

        public MainWindow()
        {
            // Initialize PDFNet.
            try
            {
                PDFNet.Initialize(PDFTronLicense.Key);
                PDFNet.EnableJavaScript(true);
                PDFNet.SetViewerCache(1000 * 1024 * 1024, true);
            }
            catch (pdftron.Common.PDFNetException ex)
            {
                MessageBox.Show(ex.GetMessage(), "Licensing Error");
            }
            catch (System.Exception ex)
            {
                MessageBox.Show(ex.Message, "Licensing Error");
            }

            PDFViewWPFSecurityHandler.MainWindow = this;
            pdftron.SDF.CreateDelegate del = new pdftron.SDF.CreateDelegate(PDFViewWPFSecurityHandler.Create);
            pdftron.SDF.SecurityManagerSingleton.Instance().RegisterSecurityHandler("Standard",
                new pdftron.SDF.SecurityDescriptor("Standard Security", del));
            InitializeComponent();

            slider_zoom.ValueChanged += slider_zoom_ValueChanged;
            txtbox_zoom.KeyDown += txtbox_zoom_KeyDown;
            txtbox_pagenum.KeyDown += txtbox_pagenum_KeyDown;
        }

        //void toolbar_footer_Loaded(object sender, RoutedEventArgs e)
        //{
        //    var overflowGrid = toolbar_footer.Template.FindName("OverflowGrid", toolbar_footer) as FrameworkElement;
        //    if (overflowGrid != null)
        //    {
        //        overflowGrid.Visibility = Visibility.Collapsed;
        //    }
        //}

        #region Methods

        public PDFViewViewer GetCurrentViewer()
        {
            PDFViewViewer view = null;
            for (int i = 0; i < PDFTab.Items.Count; ++i)
            {
                TabItem current = (TabItem)PDFTab.Items.GetItemAt(i);
                if (current.IsSelected)
                {
                    StackPanel sp = current.Header as StackPanel;
                    sp.Children[1].Visibility = Visibility.Visible;
                    Grid g = (Grid)current.Content;
                    view = (PDFViewViewer)g.Children[0];
                    _currentTab = current;
                    break;
                }
            }
            return view;
        }

        public PDFViewViewer GetViewerForTab(TabItem tab)
        {
            StackPanel sp = tab.Header as StackPanel;
            Grid g = (Grid)tab.Content;
            return (PDFViewViewer)g.Children[0];
        }

        private TabItem GetCurrentTab()
        {
            TabItem item = null;
            for (int i = 0; i < PDFTab.Items.Count; ++i)
            {
                item = (TabItem)PDFTab.Items.GetItemAt(i);
                if (item.IsSelected)
                {
                    break;
                }
            }

            return item;
        }

        public void FindText(string str, bool match_case, bool match_whole_word, bool search_up, bool reg_exp)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.FindText(str, match_case, match_whole_word, search_up, reg_exp); // Use the build in Find text dialog.
            }
            return;
        }


        public void UpdateSidewindowButtons()
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view == null)
            {
                return;
            }
            double length = view.NaviWidth;
            if (length > 10)
            {
                btn_collapse.IsEnabled = true;
            }
            else
            {
                btn_collapse.IsEnabled = false;
            }
            if (length < 200)
            {
                btn_expand.IsEnabled = true;
            }
            else
            {
                btn_expand.IsEnabled = false;
            }
        }

        public void UpdateZoomState()
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view == null)
            {
                return;
            }
            SetZoomValue(view.ZoomLevel);

        }

        // Let's a viewer tell the mainwindow that it can be closed (e.g. open URL failed).
        public void CloseViewer(PDFViewViewer viewer)
        {
            // Find the tab with the viewer
            TabItem itemToDelete = null;
            foreach (TabItem item in PDFTab.Items)
            {
                Grid g = item.Content as Grid;
                if (g != null && g.Children.Contains(viewer))
                {
                    itemToDelete = item;
                    break;
                }
            }

            if (itemToDelete != null)
            {
                itemToDelete.Content = null;
                PDFTab.Items.Remove(itemToDelete);
            }
        }

        public void SetZoomValue(double zoom)
        {
            _sliderShouldForwardValue = false;
            slider_zoom.Value = zoom;
            _sliderShouldForwardValue = true;
            string.Format("{0,0:1}", zoom);
            txtbox_zoom.Text = string.Format("{0:0.#}", zoom * 100);
        }

        #endregion

        #region Events

        public void FindTextFinishedHandler(PDFViewWPF viewer, bool found, PDFViewWPF.Selection selection)
        {
            if (_findTextDialog != null)
            {
                _findTextDialog.TextSearchFinished(viewer, found, selection);
            }
        }

        private void PDFTab_SelectionChanged(object sender, SelectionChangedEventArgs e)
        {
            PDFViewViewer newViewer = GetCurrentViewer();

            if (_currentViewer == newViewer)
            {
                return;
            }

            if (_currentViewer != null)
            {
                _currentViewer.ClearSearchSelection(true);
                _currentViewer.Deactivate();
            }
            if (_currentTab != null)
            {
                StackPanel sp = _currentTab.Header as StackPanel;
                sp.Children[1].Visibility = Visibility.Collapsed;
            }
            _currentViewer = GetCurrentViewer();
            PDFTab_UpdateCurrent();
        }

        public void PDFTab_UpdateCurrent()
        {
            if (_currentViewer != null && _currentViewer.GetPDFDoc() != null)
            {
                toolbar_footer.IsEnabled = true;
                toolbar_header2.IsEnabled = true;
                toolbar_header3.IsEnabled = true;
                btn_save.IsEnabled = true;
                btn_print.IsEnabled = true;
                txt_pagecount.Content = " / " + _currentViewer.GetPageCount().ToString();
                txtbox_pagenum.Text = _currentViewer.GetCurrentPageNumber().ToString();
                _currentViewer.UpdateToolButtonStates();
                _currentViewer.UpdatePresentationModeButtonStates();

                UpdateSidewindowButtons();
                UpdateZoomState();

                // set main window title
                this.Title = _currentViewer.GetPDFDoc().GetFileName();
                _currentViewer.Activate();
            }
            else
            {
                slider_zoom.Value = slider_zoom.Minimum;
                txtbox_zoom.Text = "";
                txtbox_pagenum.Text = "";
                txt_pagecount.Content = "";
                toolbar_footer.IsEnabled = false;
                toolbar_header2.IsEnabled = false;
                toolbar_header3.IsEnabled = false;
                btn_save.IsEnabled = false;
                btn_print.IsEnabled = false;
                this.Title = "PDFViewWPF";

                // No page presentation mode selected
                SinglePageButton.IsChecked = false;
                SingleContinuousButton.IsChecked = false;
                FacingButton.IsChecked = false;
                FacingContinuousButton.IsChecked = false;
                FacingCoverButton.IsChecked = false;
                FacingCoverContinuousButton.IsChecked = false;

                // No tool mode selected
                t_pan.IsChecked = false;
                PanToolMenuItem.IsChecked = false;

                t_annot_edit.IsChecked = false;
                AnnotationEditToolMenuItem.IsChecked = false;

                t_text_rect.IsChecked = false;

                t_text_struct.IsChecked = false;

                LineToolMenuItem.IsChecked = false;
                ArrowToolMenuItem.IsChecked = false;
                RectangleToolMenuItem.IsChecked = false;
                OvalToolMenuItem.IsChecked = false;
                InkToolMenuItem.IsChecked = false;
                TextBoxToolMenuItem.IsChecked = false;
                CalloutToolMenuItem.IsChecked = false;
                StickyNoteToolMenuItem.IsChecked = false;
                HighlightToolMenuItem.IsChecked = false;
                UnderlineToolMenuItem.IsChecked = false;
                StrikeoutToolMenuItem.IsChecked = false;
                SquigglyToolMenuItem.IsChecked = false;

            }
        }


        private void slider_zoom_ValueChanged(object sender, RoutedPropertyChangedEventArgs<double> e)
        {
            if (_sliderShouldForwardValue)
            {
                PDFViewViewer view = GetCurrentViewer();
                if (view != null)
                {
                    view.ZoomLevel = slider_zoom.Value;
                }
            }
        }

        void txtbox_pagenum_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.Key == Key.Enter && !string.IsNullOrWhiteSpace(txtbox_pagenum.Text))
            {
                int pagenum;
                if (Int32.TryParse(txtbox_pagenum.Text, out pagenum))
                {
                    PDFViewViewer view = GetCurrentViewer();
                    if (view != null)
                    {
                        view.SetCurrentPage(pagenum);
                    }
                }
            }
        }

        private void txtbox_zoom_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.Key == Key.Enter && !string.IsNullOrWhiteSpace(txtbox_zoom.Text))
            {
                double newzoom;

                if (Double.TryParse(txtbox_zoom.Text, out newzoom))
                {
                    PDFViewViewer view = GetCurrentViewer();
                    if (view != null)
                    {
                        view.ZoomLevel = newzoom / 100.0;
                    }
                }
            }
        }

        private void btnOpen_Clicked(object sender, RoutedEventArgs e)
        {
            Microsoft.Win32.OpenFileDialog dlg = new Microsoft.Win32.OpenFileDialog();
            dlg.CheckFileExists = true;
            dlg.CheckPathExists = true;
            dlg.Filter = "PDF (*.pdf)|*.pdf|All files (*.*)|*.*";
            dlg.DefaultExt = ".pdf";

            if (dlg.ShowDialog() == true)
            {
                PDFViewViewer view = new PDFViewViewer(this);
                if (view.OpenPDF(dlg.FileName))
                {
                    LoadViewer(view, dlg.SafeFileName);
                }
            }
        }

        private void btnOpenUrl_Clicked(object sender, RoutedEventArgs e)
        {
            OpenUrlDlg dlg = new OpenUrlDlg();
            dlg.Owner = this;
            if(dlg.ShowDialog() == true)
            {
                PDFViewViewer view = new PDFViewViewer(this);
                if (view.OpenURL(dlg.URL, dlg.Pass))
                {
                    LoadViewer(view, dlg.URL);
                }
            }
        }

        private void LoadViewer(PDFViewViewer view, String title)
        {
            TabItem item = new TabItem();

            // add a close button
            StackPanel tabHeaderStack = new StackPanel();
            tabHeaderStack.Orientation = Orientation.Horizontal;
            TextBlock tabLabel = new TextBlock();
            tabLabel.Text = title;
            tabHeaderStack.Children.Add(tabLabel);
            Button tabCloseButton = new Button();
            tabCloseButton.Width = 11;
            tabCloseButton.Height = 11;
            tabCloseButton.Content = "X";
            tabCloseButton.Padding = new System.Windows.Thickness(0, -3, 1, 0);
            tabCloseButton.BorderThickness = new System.Windows.Thickness(0);
            tabCloseButton.Background = Brushes.Transparent;
            tabCloseButton.Foreground = Brushes.DarkGray;
            tabCloseButton.Margin = new Thickness(5, 3, 0, 0);
            tabHeaderStack.Children.Add(tabCloseButton);
            tabCloseButton.Click += btnClose_Clicked;

            item.Header = tabHeaderStack;
            view._safeDocName = title;
            Grid grid = new Grid();
            grid.Children.Add(view);
            item.Content = grid;
            PDFTab.Items.Add(item);
            item.IsSelected = true;
        }

        private void btn_Prev_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.PrevPage();
            }
        }

        private void btn_Next_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.NextPage();
            }
        }

        private void btn_ZoomIn_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.ZoomIn();
            }
        }

        private void btn_ZoomOut_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.ZoomOut();
            }
        }

        private void btnExit_Clicked(object sender, RoutedEventArgs e)
        {
            PDFNet.Terminate();
            this.Close();
        }

        protected override void OnClosing(System.ComponentModel.CancelEventArgs e)
        {
            List<TabItem> tabsToClose = new List<TabItem>();
            if (PDFTab.HasItems)
            {
                for (int i = 0; i < PDFTab.Items.Count; ++i)
                {
                    TabItem item = (TabItem)PDFTab.Items.GetItemAt(i);
                    Grid g = (Grid)item.Content;
                    PDFViewViewer view = (PDFViewViewer)g.Children[0];
                    if (view == null) continue;
                    if (!view.CloseFile())
                    {
                        e.Cancel = true;
                        break;
                    }
                    else
                    {
                        tabsToClose.Add(item);
                    }
                }
            }
            if (e.Cancel)
            {
                foreach (TabItem tab in tabsToClose)
                {
                    PDFTab.Items.Remove(tab);
                }
            }
            base.OnClosing(e);
        }

        private void btn_LastPage_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.LastPage();
            }
        }

        private void btn_FirstPage_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.FirstPage();
            }
        }

        private void btn_RotateCCW_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.RotateCounterClockwise();
            }
        }

        private void btn_RotateCW_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.RotateClockwise();
            }
        }

        private void FitWidth_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.FitWidth();
            }
        }

        private void FitPageClicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.FitPage();
            }
        }

        private void SinglePage_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.PageSingle();
            }
        }

        private void SingleContinuous_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.PageSingleContinuous();
            }
        }

        private void Facing_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.PageFacing();
            }
        }

        private void FacingContinuous_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.PageFacingContinuous();
            }
        }

        private void FacingCoverContinuous_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.PageFacingCoverContinous();
            }
        }

        private void FacingCover_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.PageFacingCover();
            }
        }
        
        private void btnPrint_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                PrintDialog pDlg = new PrintDialog();
                pDlg.UserPageRangeEnabled = true;
                Nullable<Boolean> result = pDlg.ShowDialog();
                if (result == true)
                {
                    PrintQueue pq = pDlg.PrintQueue;
                    PrintTicket pt = pDlg.PrintTicket;
                    PageSet ps = new PageSet(pDlg.PageRange.PageFrom, pDlg.PageRange.PageTo);
                    if (pDlg.PageRangeSelection == PageRangeSelection.AllPages)
                        ps = new PageSet(1, view.GetPageCount());
                    PrinterMode pm = new PrinterMode();
                    pm.SetScaleType(PrinterMode.ScaleType.e_ScaleType_ReduceToOutputPage);
                    if (view.GetRotation() == pdftron.PDF.Page.Rotate.e_90 || view.GetRotation() == pdftron.PDF.Page.Rotate.e_270)
                        pm.SetOrientation(PrinterMode.Orientation.e_Orientation_Landscape);
                    else
                        pm.SetOrientation(PrinterMode.Orientation.e_Orientation_Portrait);
                    BackgroundWorker bgPrintWorker = new BackgroundWorker();
                    bgPrintWorker.DoWork += BackgroundPrintTask;
                    bgPrintWorker.RunWorkerCompleted += bgPrintWorker_RunWorkerCompleted;
                    bgPrintWorker.RunWorkerAsync(new PrinterArguments(view.GetPDFDoc(), pq.Name, view.GetPDFDoc().GetFileName(), ps, pm, view.Current_View.GetOCGContext()));
                    
                }
            }
        }

        void bgPrintWorker_RunWorkerCompleted(object sender, RunWorkerCompletedEventArgs e)
        {
            if (e.Error != null)
            {
                MessageBox.Show(string.Format("Printing failed: {0}{0}{1}", Environment.NewLine, e.Error.Message));
            }
        }

        void BackgroundPrintTask(object sender, DoWorkEventArgs e)
        {

            PrinterArguments printerArgs = (PrinterArguments)e.Argument;
            try
            {
                printerArgs.DocToPrint.LockRead();
                Print.StartPrintJob(printerArgs.DocToPrint, printerArgs.PrinterName, printerArgs.DocToPrint.GetFileName(),
                    "", printerArgs.PageSet, printerArgs.PrinterMode, printerArgs.OCG);
            }
            catch (Exception ex)
            {
                throw new Exception(ex.Message);
            }
            finally
            {
                printerArgs.DocToPrint.UnlockRead();
            }
        }
        
        private void btnClose_Clicked(object sender, RoutedEventArgs e)
        {
            TabItem tab = null;
            PDFViewViewer view = null;
            if (sender is Button && (sender as Button).Parent is StackPanel)
            {
                // Get the TabItem that owns the stackPanel object
                tab = (TabItem)((sender as Button).Parent as StackPanel).Parent;
                view = GetViewerForTab(tab);
            }
            else
            {
                view = GetCurrentViewer();
                tab = GetCurrentTab();
            }
            if (view != null)
            {
                if (view.CloseFile())
                {
                    if (tab != null)
                    {
                        PDFTab.Items.Remove(tab);
                    }
                }
            }
        }

        private void btnSave_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                PDFDoc doc = view.GetPDFDoc();
                if (doc != null)
                {
                    if (String.IsNullOrEmpty(doc.GetFileName()))
                    {
                        btnSaveAs_Clicked(sender, e);
                    }
                    else
                    {
                        view.Save(doc.GetFileName());
                    } 
                }
            }
        }

        private void btnSaveAs_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                string newname = view.SaveAs();
                if (newname != null)
                {
                    TabItem item = GetCurrentTab();
                    StackPanel sp = item.Header as StackPanel;
                    TextBlock tb = sp.Children[0] as TextBlock;
                    tb.Text = newname;
                    view._safeDocName = newname;
                    this.Title = view.GetPDFDoc().GetFileName();
                }
            }
        }

        private void btnExpand_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.OpenNavibar();
            }
        }

        private void btnCollapsed_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.CloseNavibar();
            }
        }

        private void btnSelectAll_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.SelectAll();
            }
        }

        private void btnDeselectAll_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.DeselectAll();
            }
        }

        #endregion Events

        #region ToolMode

        //////////////////////////////////////////////////////////////////////////
        // Note: Tools that create or add annotations can be created either as 
        // persistent or not. If not persistent any shape creation tool will revert 
        // back to the pan tool when done.
        // If persistent, any number of shapes can be drawn, and the tool has to 
        // be closed explicitly (by creating another tool).
        // This can be changed by setting the 3rd parameter of ToolManager.CreateTool
        // to true (persistent) or false (not persistent).
        //////////////////////////////////////////////////////////////////////////
        private void PanTool_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.ToolManager.CreateTool(pdftron.PDF.Tools.ToolManager.ToolType.e_pan, null);
            }

        }

        private void TextSelectStruct_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.ToolManager.CreateTool(pdftron.PDF.Tools.ToolManager.ToolType.e_text_select, null);
            }
        }

        private void TextSelectRect_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.ToolManager.CreateTool(pdftron.PDF.Tools.ToolManager.ToolType.e_text_select_rectangular, null);
            }
        }

        private void AnnotationEditTool_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.ToolManager.CreateTool(pdftron.PDF.Tools.ToolManager.ToolType.e_annot_edit, null);
            }

        }

        private void LineTool_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.ToolManager.CreateTool(pdftron.PDF.Tools.ToolManager.ToolType.e_line_create, null, true);
            }
        }

        private void PolylineTool_Click(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.ToolManager.CreateTool(pdftron.PDF.Tools.ToolManager.ToolType.e_polyline_create, null, true);
            }
        }

        private void PolygonTool_Click(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.ToolManager.CreateTool(pdftron.PDF.Tools.ToolManager.ToolType.e_polygon_create, null, true);
            }
        }

        private void ArrowTool_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.ToolManager.CreateTool(pdftron.PDF.Tools.ToolManager.ToolType.e_arrow_create, null, true);
            }

        }

        private void OvalTool_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.ToolManager.CreateTool(pdftron.PDF.Tools.ToolManager.ToolType.e_oval_create, null, true);
            }
        }

        private void RectangleTool_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.ToolManager.CreateTool(pdftron.PDF.Tools.ToolManager.ToolType.e_rect_create, null, true);
            }
        }

        private void InkTool_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.ToolManager.CreateTool(pdftron.PDF.Tools.ToolManager.ToolType.e_ink_create, null, true);

                // You can create a multi-path ink annotation as follows:
                // This means that you can have an annotation with multiple strokes.
                //pdftron.PDF.Tools.FreehandCreate fhc = view.ToolManager.CurrentTool as pdftron.PDF.Tools.FreehandCreate;
                //fhc.CreateMultiPathAnnotation = true;

                // You can then use the following 2 functions to save or discard the current shape
                //fhc.SaveShape();
                //fhc.DiscardShape();

                // Currently, the FreeHandCreate tool is set up so that the key 'y' will save the drawing and the key 
                // 'n' will discard it. Also, on the right click context menu, there's a save and discard option
            }

        }

        private void HighlightTool_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.ToolManager.CreateTool(pdftron.PDF.Tools.ToolManager.ToolType.e_text_highlight, null, true);
            }

        }

        private void UnderlineTool_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.ToolManager.CreateTool(pdftron.PDF.Tools.ToolManager.ToolType.e_text_underline, null, true);
            }

        }

        private void StrikeoutTool_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.ToolManager.CreateTool(pdftron.PDF.Tools.ToolManager.ToolType.e_text_strikeout, null, true);
            }

        }

        private void SquigglyTool_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.ToolManager.CreateTool(pdftron.PDF.Tools.ToolManager.ToolType.e_text_squiggly, null, true);
            }

        }

        private void FreeTextToolTool_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.ToolManager.CreateTool(pdftron.PDF.Tools.ToolManager.ToolType.e_text_annot_create, null, true);
            }
        }

        private void CalloutTool_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.ToolManager.CreateTool(pdftron.PDF.Tools.ToolManager.ToolType.e_callout_create, null, true);
            }
        }
        
        
        private void StickyNoteTool_Clicked(object sender, RoutedEventArgs e)
        {
            PDFViewViewer view = GetCurrentViewer();
            if (view != null)
            {
                view.ToolManager.CreateTool(pdftron.PDF.Tools.ToolManager.ToolType.e_sticky_note_create, null, true);
            }

        }

        #endregion ToolMode
    }
}
