'
' Copyright (c) 2001-2025 by Apryse Software Inc. All Rights Reserved.
'
Imports System

Imports pdftron
Imports pdftron.Common
Imports pdftron.SDF
Imports pdftron.PDF


'-----------------------------------------------------------------------------------
' The sample illustrates how to work with PDF page labels.
'
' PDF page labels can be used to describe a page. This is used to 
' allow for non-sequential page numbering or the addition of arbitrary 
' labels for a page (such as the inclusion of Roman numerals at the 
' beginning of a book). PDFNet PageLabel object can be used to specify 
' the numbering style to use (for example, upper- or lower-case Roman, 
' decimal, and so forth), the starting number for the first page,
' and an arbitrary prefix to be pre-appended to each number (for 
' example, "A-" to generate "A-1", "A-2", "A-3", and so forth.)
'-----------------------------------------------------------------------------------
Module PageLabelsTestVB
	Dim pdfNetLoader As PDFNetLoader
	Sub New()
		pdfNetLoader = pdftron.PDFNetLoader.Instance()
	End Sub

	' Relative path to the folder containing test files.
	Dim input_path As String = "../../../../TestFiles/"
	Dim output_path As String = "../../../../TestFiles/Output/"

	Sub Main()

		PDFNet.Initialize(PDFTronLicense.Key)
		Try
			'-----------------------------------------------------------
			' Example 1: Add page labels to an existing or newly created PDF
			' document.
			'-----------------------------------------------------------
			Using doc As PDFDoc = New PDFDoc(input_path + "newsletter.pdf")
				doc.InitSecurityHandler()

				' Create a page labeling scheme that starts with the first page in 
				' the document (page 1) and is using uppercase roman numbering 
				' style. 
				doc.SetPageLabel(1, PageLabel.Create(doc.GetSDFDoc(), PageLabel.Style.e_roman_uppercase, "My Prefix ", 1))

				' Create a page labeling scheme that starts with the fourth page in 
				' the document and is using decimal arabic numbering style. 
				' Also the numeric portion of the first label should start with number 
				' 4 (otherwise the first label would be "My Prefix 1"). 
				Dim L2 As PageLabel = PageLabel.Create(doc.GetSDFDoc(), PageLabel.Style.e_decimal, "My Prefix ", 4)
				doc.SetPageLabel(4, L2)

				' Create a page labeling scheme that starts with the seventh page in 
				' the document and is using alphabetic numbering style. The numeric 
				' portion of the first label should start with number 1. 
				Dim L3 As PageLabel = PageLabel.Create(doc.GetSDFDoc(), PageLabel.Style.e_alphabetic_uppercase, "My Prefix ", 1)
				doc.SetPageLabel(7, L3)

				doc.Save(output_path + "newsletter_with_pagelabels.pdf", SDF.SDFDoc.SaveOptions.e_linearized)
				Console.WriteLine("Done. Result saved in newsletter_with_pagelabels.pdf...")
			End Using
		Catch ex As PDFNetException
			Console.WriteLine(ex.Message)
		Catch ex As Exception
			MsgBox(ex.Message)
		End Try

		'-----------------------------------------------------------
		' Example 2: Read page labels from an existing PDF document.
		'-----------------------------------------------------------
		Try
			Using doc As PDFDoc = New PDFDoc(output_path + "newsletter_with_pagelabels.pdf")
				doc.InitSecurityHandler()

				Dim label As PageLabel
				Dim page_num As Integer = doc.GetPageCount()

				Dim i As Integer
				For i = 1 To page_num
					Console.Write("Page number: {0}", i)
					label = doc.GetPageLabel(i)
					If label.IsValid() Then
						Console.WriteLine(" Label: {0}", label.GetLabelTitle(i))
					Else
						Console.WriteLine(" No Label.")
					End If
				Next i
			End Using
		Catch ex As PDFNetException
			Console.WriteLine(ex.Message)
		Catch ex As Exception
			MsgBox(ex.Message)
		End Try

		'-----------------------------------------------------------
		' Example 3: Modify page labels from an existing PDF document.
		'-----------------------------------------------------------
		Try
			Using doc As PDFDoc = New PDFDoc(output_path + "newsletter_with_pagelabels.pdf")
				doc.InitSecurityHandler()

				'Remove the alphabetic labels from example 1.
				doc.RemovePageLabel(7)

				' Replace the Prefix in the decimal lables (from example 1).
				Dim label As PageLabel = doc.GetPageLabel(4)
				If (label.IsValid()) Then
					label.SetPrefix("A")
					label.SetStart(1)
				End If

				' Add a new label
				Dim new_label As PageLabel = PageLabel.Create(doc.GetSDFDoc(), PageLabel.Style.e_decimal, "B", 1)
				doc.SetPageLabel(10, new_label)		   ' starting from page 10.

				doc.Save(output_path + "newsletter_with_pagelabels_modified.pdf", SDF.SDFDoc.SaveOptions.e_linearized)
				Console.WriteLine("Done. Result saved in newsletter_with_pagelabels_modified.pdf...")

				Dim page_num As Integer = doc.GetPageCount()
				Dim i As Integer
				For i = 1 To page_num
					Console.Write("Page number: {0}", i)
					label = doc.GetPageLabel(i)
					If (label.IsValid()) Then
						Console.WriteLine(" Label: {0}", label.GetLabelTitle(i))
					Else
						Console.WriteLine(" No Label.")
					End If
				Next i
			End Using
		Catch ex As PDFNetException
			Console.WriteLine(ex.Message)
		Catch ex As Exception
			MsgBox(ex.Message)
		End Try

		'-----------------------------------------------------------
		' Example 4: Delete all page labels in an existing PDF document.
		'-----------------------------------------------------------
		Try
			Using doc As PDFDoc = New PDFDoc(output_path + "newsletter_with_pagelabels.pdf")
				doc.GetRoot().Erase("PageLabels")
				' ...
			End Using
		Catch ex As PDFNetException
			Console.WriteLine(ex.Message)
		Catch ex As Exception
			MsgBox(ex.Message)
		End Try
		PDFNet.Terminate()
	End Sub
End Module
